
:setvar DatabaseName "Registry"
:setvar MajorReleaseNumber "1"
:setvar MinorReleaseNumber "0"
:setvar PointReleaseNumber "51"
:setvar DeployVersion "20150909.1"

PRINT ' 
------------------------------------------------------------------------------------------------
Registry.AHOBPR.Database.Build.1.0.51.sql

Version Release $(MajorReleaseNumber).$(MinorReleaseNumber).$(PointReleaseNumber)

Build $(DeployVersion)

------------------------------------------------------------------------------------------------
 ';
PRINT 'Started ' + CAST(GETDATE() as varchar(20)) + '.';
PRINT ' 
*********************************************************
NOTE: This script must be run with SQLCMD Mode turned on.
*********************************************************

';
GO
:on error exit
GO
:setvar __IsSqlCmdEnabled "True"
/*
Detect SQLCMD mode and disable script execution if SQLCMD mode is not supported.
To re-enable the script after enabling SQLCMD mode, execute the following:
SET NOEXEC OFF; 
*/
GO
IF N'$(__IsSqlCmdEnabled)' NOT LIKE N'True'
    BEGIN
        PRINT N'SQLCMD mode must be enabled to successfully execute this script.';
        SET NOEXEC ON;
    END

GO

USE [$(DatabaseName)]
GO

PRINT 'DROP TRIGGER TRG_REGISTRANT_JSON_INSERT...';
GO

IF  EXISTS (SELECT * FROM sys.triggers WHERE object_id = OBJECT_ID(N'[AHOBPR].[TRG_REGISTRANT_JSON_INSERT]'))
DROP TRIGGER [AHOBPR].[TRG_REGISTRANT_JSON_INSERT]
GO

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[AHOBPR].[SP_JsonGetCurrent]') AND type in (N'P', N'PC'))
DROP PROCEDURE [AHOBPR].[SP_JsonGetCurrent]
GO

PRINT 'CREATE PROCEDURE SP_JsonGetCurrent...';
GO

CREATE PROCEDURE [AHOBPR].[SP_JsonGetCurrent]
@ObjectType VARCHAR(100),
 @ObjectId VARCHAR(256)
AS

BEGIN

DECLARE @CurrentId INT;
        
      -- Get the Most Current JSON ID for the Type and ID
      SELECT @CurrentId = MAX([REGISTRANT_JSON_ID]) 
        FROM [AHOBPR].[REGISTRANT_JSON] WITH (READPAST)
       WHERE [OBJECT_TYPE] = @ObjectType  
         AND [OBJECT_ID] = @ObjectId;
         
      -- Get the record   
      SELECT * 
        FROM [AHOBPR].[REGISTRANT_JSON] WITH (NOLOCK)
       WHERE [REGISTRANT_JSON_ID] = @CurrentId;

END
-- EXECUTE AHOBPR.SP_JsonGetCurrent 5187

GO

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[AHOBPR].[sp_getJsonQueue]') AND type in (N'P', N'PC'))
DROP PROCEDURE [AHOBPR].[SP_GetJsonQueue]
GO

PRINT 'CREATE PROCEDURE SP_GetJsonQueue...';
GO

CREATE PROCEDURE [AHOBPR].[SP_GetJsonQueue]
AS
/*
SP_GetJsonQueue 
Get all the items that need to be processed by JSON_QUEUE_IDs:
a.	Delete the previous/old/duplicated records from REGISTRANT_JSON table
b.	Insert the new records to JSON_QUEUE table
c.	Order the JSON_QUEUE_IDs to be processed by object_type in the following order:  USER, INFO, RESPONSE 
d.	Ignore any REGISTRANT_JSON records that are less than 30 minutes old
*/

BEGIN

  -- DELETE any duplicate REGISTRANT_JSON rows
  WITH 
    RankedRow AS 
    (
      SELECT [REGISTRANT_JSON_ID]
            ,RANK() OVER(PARTITION BY [OBJECT_TYPE], [OBJECT_ID]
                             ORDER BY [REGISTRANT_JSON_ID] DESC) AS RowRank
        FROM 
            [AHOBPR].[REGISTRANT_JSON] WITH (READPAST)--don't bother with records that are locked
    )
  , Duplicate AS
    (
       SELECT   [REGISTRANT_JSON_ID] 
         FROM   RankedRow
         WHERE  RowRank > 1
    )
  DELETE 
    FROM [AHOBPR].[REGISTRANT_JSON] WITH (READPAST)--don't bother with records that are locked
    WHERE EXISTS  ( SELECT * 
                    FROM  Duplicate
                    WHERE Duplicate.REGISTRANT_JSON_ID = REGISTRANT_JSON.REGISTRANT_JSON_ID
                  );
                     
  -- Insert the new records in the JSON_QUEUE     
  -- Ignore any REGISTRANT_JSON records that are less than 30 minutes old
  DECLARE @ThirtyMinutesAgo DATETIME2(0) = DATEADD(MINUTE, -30, GETDATE());
  WITH 
    RankedRow AS --we just deleted duplicates above, but since we used READPAST and since there is no
                 --guarantee that new dup has not been added since we take the precaution of ranking
                 --them again.
    ( SELECT *
            ,RANK() OVER(PARTITION BY [OBJECT_TYPE], [OBJECT_ID]
                             ORDER BY [REGISTRANT_JSON_ID] DESC) AS RowRank
       FROM [AHOBPR].[REGISTRANT_JSON] WITH (READPAST) --don't bother with records that are locked
       WHERE NOT EXISTS ( SELECT  *
                          FROM    AHOBPR.JSON_QUEUE AS Q
                          WHERE   Q.REGISTRANT_JSON_ID = [REGISTRANT_JSON].REGISTRANT_JSON_ID
                        )
        AND UPDATED < @ThirtyMinutesAgo
    )
  , Src AS
    ( SELECT [REGISTRANT_JSON_ID]
            ,[CREATED]
            ,[UPDATED]
            ,[CREATEDBY]
            ,[UPDATEDBY]
      FROM  RankedRow
      WHERE RowRank = 1      
    )
    MERGE 
      INTO  [AHOBPR].[JSON_QUEUE] Tgt
      USING Src
        ON  Tgt.[REGISTRANT_JSON_ID] = Src.[REGISTRANT_JSON_ID]
       WHEN NOT MATCHED
       THEN INSERT  
            ( [REGISTRANT_JSON_ID]
            , [CREATED]
            , [UPDATED]
            , [CREATEDBY]
            , [UPDATEDBY]
            )
            VALUES
            (  Src.[REGISTRANT_JSON_ID]
              ,Src.[CREATED]
              ,Src.[UPDATED]
              ,Src.[CREATEDBY]
              ,Src.[UPDATEDBY]
            );
            
-- Get the IDs of the queued JSON objects to be parsed
-- Order the JSON_QUEUE_IDs to be processed by object_type in the following order:  USER, INFO, RESPONSE 
  SELECT  Q.JSON_QUEUE_ID
  FROM    
          AHOBPR.JSON_QUEUE AS Q WITH (READPAST)
  INNER
  JOIN    AHOBPR.REGISTRANT_JSON AS J WITH (READPAST)
    ON    Q.REGISTRANT_JSON_ID = J.REGISTRANT_JSON_ID
  WHERE  ISNULL(Q.PROCESS_STATUS,'failed') = 'failed'
    AND   J.OBJECT_TYPE     != 'form'
  ORDER
    BY    CASE J.OBJECT_TYPE  WHEN 'user' THEN 1
                              WHEN 'info' THEN 2
                              WHEN 'responses' THEN 3
                              ELSE 99 --should not happen
          END
END  -- 12 rows
-- EXECUTE [AHOBPR].[SP_GetJsonQueue]
--select * from ahobpr.json_queue where json_queue_id = 11163
GO

PRINT 'ALTER PROCEDURE SP_UpdateFromCDW...';
GO
-- =============================================
-- Author:		Bruce Clegg
-- Create date: 6/12/2015
-- Description:	Called after loading CDW Staging tables
-- 08/12/2015 - Set Gender to MALE and FEMALE
-- 09/08/2015 - Only set Gender and ICN if Registrant value is NULL
-- =============================================
ALTER PROCEDURE [AHOBPR].[SP_UpdateFromCDW]
AS
BEGIN

  DECLARE @Now DATETIME2(0) = GETDATE();
  DECLARE @ProcName VARCHAR(30) = 'AHOBPR.SP_UpdateFromCDW';

  PRINT 'Set Institution ID in Staged Patient';
  
  WITH Src
  AS(
  SELECT REGISTRANT_ID
	    ,PreferredInstitutionCode
	    ,Inst.ID
    FROM [AHOBPR].[STAGED_PATIENT] Stg
    JOIN AHOBPR.VIEW_INSTITUTION Inst ON Inst.STATIONNUMBER = Stg.PreferredInstitutionCode
    )
    MERGE INTO [AHOBPR].[STAGED_PATIENT] AS Tgt
    USING Src
	   ON Tgt.REGISTRANT_ID = Src.REGISTRANT_ID
    WHEN MATCHED
    THEN UPDATE
		    SET Tgt.STD_INSTITUTION_ID = Src.ID;
  
  PRINT 'Update Registrant from staged patient...';

  WITH Src
  AS (
    SELECT reg.REGISTRANT_ID
          ,ISNULL(reg.GENDER,CASE pat.GENDER WHEN 'F' THEN 'FEMALE' 
 		                                       WHEN 'M' THEN 'MALE' END) as GENDER 
          ,pat.DECEASED
          ,pat.DateOfDeath
          ,pat.RaceText
          ,pat.MaritalStatus
          ,pat.EligibilityStatus
          ,pat.Eligibility
          ,ISNULL(reg.ICN,pat.PatientICN) AS PatientICN
          ,pat.EnrollmentStatus
          ,pat.PreferredInstitutionCode
          ,pat.EnrollmentDate
          ,pat.STD_INSTITUTION_ID
      FROM [AHOBPR].[STAGED_PATIENT] pat
      JOIN [AHOBPR].[REGISTRANT] reg ON reg.REGISTRANT_ID = pat.REGISTRANT_ID
      )
      MERGE INTO [AHOBPR].[REGISTRANT] AS Tgt
      USING Src
	     ON Tgt.REGISTRANT_ID = Src.REGISTRANT_ID
      WHEN MATCHED
      THEN UPDATE
		      SET Tgt.STD_INSTITUTION_ID = Src.STD_INSTITUTION_ID
 		         ,Tgt.GENDER = Src.GENDER    
 		         ,Tgt.MARITAL_STATUS = Src.[MaritalStatus] 
 		         ,Tgt.RACE = Src.[RaceText]  
 		         ,Tgt.DECEASED_FLAG = CONVERT(BIT,(CASE Src.[DECEASED] WHEN 'Y' THEN 1 ELSE 0 END))
 		         ,Tgt.DEATH_DATE = Src.[DateOfDeath]    
 		         ,Tgt.ELIGIBILITY_STATUS = Src.[EligibilityStatus]     		        
 		         ,Tgt.ELIGIBILITY = Src.[Eligibility]    
 		         ,Tgt.ENROLLMENT_STATUS = Src.[EnrollmentStatus]    
 		         ,Tgt.ENROLLED_DATE = Src.[EnrollmentDate]       
 		         ,Tgt.ETL_UPDATED = @Now;
  		    
  PRINT 'Delete health factors for encounters being updated';
  DELETE 
  FROM AHOBPR.REGISTRANT_HEALTH_FACTOR
  WHERE EXISTS(SELECT REGISTRANT_VISIT_ID 
                 FROM AHOBPR.REGISTRANT_VISIT
                 JOIN AHOBPR.STAGED_VISIT ON STAGED_VISIT.REGISTRANT_ID = REGISTRANT_VISIT.REGISTRANT_ID
                WHERE REGISTRANT_VISIT.REGISTRANT_VISIT_ID = REGISTRANT_HEALTH_FACTOR.REGISTRANT_VISIT_ID
                  AND STAGED_VISIT.EVALUATION_FLAG = 'Y');
                  
  PRINT 'Delete encouters being updated';

  DELETE 
  FROM AHOBPR.REGISTRANT_VISIT
  WHERE EVALUATION_FLAG = 'Y'
    AND EXISTS(SELECT REGISTRANT_ID FROM AHOBPR.STAGED_VISIT
                WHERE STAGED_VISIT.REGISTRANT_ID = REGISTRANT_VISIT.REGISTRANT_ID
                AND STAGED_VISIT.EVALUATION_FLAG = 'Y');

  PRINT 'Delete recent visits being updated';              

  DELETE 
  FROM AHOBPR.REGISTRANT_VISIT
  WHERE EVALUATION_FLAG = 'N'
    AND EXISTS(SELECT REGISTRANT_ID FROM AHOBPR.STAGED_VISIT
                WHERE STAGED_VISIT.REGISTRANT_ID = REGISTRANT_VISIT.REGISTRANT_ID
                AND STAGED_VISIT.EVALUATION_FLAG = 'N');               		      
  
  PRINT 'Set Institution ID in Staged VISIT';
  
  WITH Src
  AS(
  SELECT Stg.VisitSID
        ,Stg.INSTITUTION_CODE
        ,Inst.ID
    FROM [AHOBPR].[STAGED_VISIT] Stg
    JOIN AHOBPR.VIEW_INSTITUTION Inst ON Inst.STATIONNUMBER = Stg.INSTITUTION_CODE
    )
    MERGE INTO [AHOBPR].[STAGED_VISIT] AS Tgt
    USING Src
	   ON Tgt.VisitSID = Src.VisitSID
    WHEN MATCHED
    THEN UPDATE
		    SET Tgt.STD_INSTITUTION_ID = Src.ID;		    
		    
  PRINT 'Insert encounters and visits...';              

  INSERT INTO [AHOBPR].[REGISTRANT_VISIT]
             ([REGISTRANT_ID]
             ,[VISIT_DATE_TIME]
             ,[STD_INSTITUTION_ID]
             ,[VISTA_EncounterID]
             ,[VISTA_Sta3n]
             ,[VISTA_EligibilityStatus]
             ,[EVALUATION_FLAG]
             ,[APPOINTMENT_TYPE]
             ,[STOP_CODE]
             ,[CREATEDBY]
             ,[UPDATEDBY]
             ,[CREATED]
             ,[UPDATED])
      SELECT [REGISTRANT_ID]
            ,[VISIT_DATE_TIME]
            ,[STD_INSTITUTION_ID]
            ,[VISTA_EncounterID]
            ,[VISTA_Sta3n]
            ,[VISTA_Eligibility]
            ,[EVALUATION_FLAG]
            ,[APPOINTMENT_TYPE]
            ,[STOP_CODE]
            ,@ProcName
            ,@ProcName
            ,@Now
            ,@Now
    FROM [AHOBPR].[STAGED_VISIT];		    
 
  PRINT 'Set Standard Health Factors ID from STD_HEALTH_FACTOR...';  
    
  WITH Src
   AS(
      SELECT [CDW_VISIT_SID]
            ,[REGISTRANT_ID]
            ,std.[STD_HEALTH_FACTOR_ID]
            ,[HEALTH_FACTOR_TEXT]      
        FROM [AHOBPR].[STAGED_HEALTH_FACTOR] staged
        JOIN AHOBPR.STD_HEALTH_FACTOR std ON std.NAME = staged.HEALTH_FACTOR_TEXT
    )
    MERGE INTO AHOBPR.[STAGED_HEALTH_FACTOR] AS Tgt
    USING Src  ON Tgt.[CDW_VISIT_SID] = Src.[CDW_VISIT_SID]
    WHEN MATCHED
             AND Tgt.[REGISTRANT_ID] = Src.[REGISTRANT_ID]
             AND Tgt.[HEALTH_FACTOR_TEXT] = Src.[HEALTH_FACTOR_TEXT]
    THEN UPDATE
		    SET Tgt.[STD_HEALTH_FACTOR_ID] = Src.[STD_HEALTH_FACTOR_ID];
 
   PRINT 'Insert Health Factors...';  
   
  INSERT INTO [AHOBPR].[REGISTRANT_HEALTH_FACTOR]
             ([REGISTRANT_VISIT_ID]
             ,[STD_HEALTH_FACTOR_ID]
             ,[ACTIVE_FLAG]
             ,[HEALTH_FACTOR_TEXT]
             ,[COMMENTS]
             ,[CREATEDBY]
             ,[UPDATEDBY]
             ,[CREATED]
             ,[UPDATED])          
      SELECT visit.REGISTRANT_VISIT_ID
            ,hf.[STD_HEALTH_FACTOR_ID]
            ,1
            ,hf.[HEALTH_FACTOR_TEXT]
            ,hf.[COMMENTS]
            ,@ProcName    
            ,@ProcName
            ,@Now
            ,@Now     
    FROM [AHOBPR].[STAGED_HEALTH_FACTOR] hf -- 295
    JOIN AHOBPR.STAGED_VISIT stage ON stage.VisitSID = hf.CDW_VISIT_SID
    JOIN AHOBPR.REGISTRANT_VISIT visit ON visit.VISTA_EncounterID = stage.VISTA_EncounterID 

END;
-- EXECUTE [AHOBPR].[SP_UpdateFromCDW]; 

GO

PRINT 'ALTER PROCEDURE SP_ReportStatusByLocation...';
GO

-- =============================================
-- Author:		Deborah Barr
-- Create date: November 12, 2014
-- Description:	Used for the AHOBPR Status by Location Report

-- Modification log
-- 20150114 - Sherry Wang: Changed query for @InstitutionId and @VisnId
-- 20150202 - Change state to use state codes, removed unneeded tables and joins, Removed Distinct
-- 20150219 - Change state to include territories and Military Bases
-- 20150330 - Bruce Clegg - Added VisnId value of NLA to find Null locations
-- 20150427 - Bruce Clegg - Change to LocationType and Key, for both Assigned and Evaluated Locations
-- 20150722 - Bruce Clegg - Fix bug with @EvaluatedLocationType = 'Base'
-- 20150909 - Bruce Clegg - Rewrite as Dynamic SQL for Performance
-- =============================================
ALTER PROCEDURE [AHOBPR].[SP_ReportStatusByLocation]
       @state varchar(8000) = NULL, -- Multiple states, comma-separated values when not NULL 
       @status varchar(1000) = NULL, -- Multiple statues, comma-separated values when not NULL        
       @AssignedLocationType  VARCHAR(20) = NULL,  -- 'VISN', 'Base', 'Institution', 'NLA' = No Location Assigned
       @AssignedLocationKey   VARCHAR(50) = NULL,  -- 'V16', 'V01'    1000109,  1000089
       @EvaluatedLocationType VARCHAR(20) = NULL,  -- 'VISN', 'Base', 'Institution', 'NLA' = No Location Assigned
       @EvaluatedLocationKey  VARCHAR(50) = NULL  -- 'V16', 'V01'    1000109,  1000089
AS
BEGIN
  DECLARE @Today DATETIME2 = GETDATE();

  DECLARE @Query VARCHAR(8000);
 
   --Temp table concatnate flags for Registrants with multiple flags
  CREATE TABLE #state
  (
     [Registrant_Id] INT,
     [state] VARCHAR(8000)
  ) 

  SET @state = REPLACE(@state, 'Foreign Country/Other', 'FM,MH,PW,UM,US,CANAD,MX,EU,PI,FG,OT,UN')

  INSERT INTO #state ([Registrant_Id], [state])
  -- Load Table
  SELECT a.REGISTRANT_ID, 
        STUFF((SELECT ', ' + b.STATE 
                 FROM AHOBPR.REGISTRANT_ADDRESS b 
                WHERE a.REGISTRANT_ID = b.REGISTRANT_ID 
                FOR XML PATH('')), 1, 1, '')  [STATE]
    FROM AHOBPR.REGISTRANT a
   GROUP BY a.REGISTRANT_ID;
 
 
  SET @Query = 'SELECT  reg.REGISTRANT_ID ' +
		                   ',(reg.LAST_NAME + '', '' + reg.FIRST_NAME) AS Name ' +
		                   ',b.REGISTRANT_STATUS AS Status ' +
		                   ',DATEDIFF(day, reg.REGISTRANT_STATUS_LAST_UPDATED_DATE, ' +
		                   CONVERT(VARCHAR(10),@Today,121) + ') As Days ' +
		                   ',s.STATE AS State ' +
		                   ',RIGHT(inst.VisnID, 2)  AS VISN ' +
		                   ',inst.Name AS Facility ' +
		                   ',reg.BIRTH_DATE AS BirthDate ' +
                       ',CONVERT(VARCHAR(10), reg.BIRTH_DATE, 101) AS BirthDateDisplay ' +
	               'FROM AHOBPR.REGISTRANT reg ' +
		             'JOIN AHOBPR.STD_REGISTRANT_STATUS b ' + 
		               'ON reg.STD_REGISTRANT_STATUS_ID = b.STD_REGISTRANT_STATUS_ID ' +
	          'LEFT JOIN AHOBPR.VIEW_FACILITY_LOOKUP inst ' +
		               'ON inst.ID = reg.STD_INSTITUTION_ID ' +
		        'LEFT JOIN #state s ' +
		               'ON reg.REGISTRANT_ID = s.Registrant_Id ' +
	              'WHERE  1 = 1 ';

	              
  IF ISNULL(@state,'') > ''
    BEGIN
      SET @state = REPLACE(@State,',',''',''');
      SET @Query = @Query + ' AND EXISTS(SELECT DISTINCT addr.REGISTRANT_ID ' + 
                                          'FROM AHOBPR.REGISTRANT_ADDRESS addr ' + 
                                         'WHERE addr.REGISTRANT_ID = reg.REGISTRANT_ID ' + 
                                           'AND addr.STATE  IN ( ''' + @state + ''') ) ';
    END;

  IF ISNULL(@status,'') > '' 
	 	 SET @Query = @Query + ' AND reg.STD_REGISTRANT_STATUS_ID IN (' + @status + ') ';
	 	 
  IF @AssignedLocationType = 'NLA' 
     SET @Query = @Query + ' AND reg.STD_INSTITUTION_ID IS NULL ';

  IF @AssignedLocationType = 'VISN' 
     SET @Query = @Query + ' AND EXISTS(SELECT * FROM AHOBPR.VIEW_FACILITY_LOOKUP inst ' +
                                        'WHERE inst.ID = reg.STD_INSTITUTION_ID ' +
                                          'AND inst.VisnID = ''' + @AssignedLocationKey + ''' ) ';

  IF @AssignedLocationType = 'Base' 
     SET @Query = @Query + 'AND EXISTS(SELECT * FROM AHOBPR.VIEW_FACILITY_LOOKUP inst ' +
                                       'WHERE inst.ID = reg.STD_INSTITUTION_ID ' +
                                         'AND inst.BASE_ID = ' + @AssignedLocationKey + ') ';

  IF @AssignedLocationType = 'Institution' 
     SET @Query = @Query + 'AND reg.STD_INSTITUTION_ID = ' + @AssignedLocationKey + ' ';            

  IF @EvaluatedLocationType = 'NLA' 
      SET @Query = @Query + 'AND NOT EXISTS( SELECT * FROM AHOBPR.REGISTRANT_VISIT eval ' +
                                          '   WHERE eval.REGISTRANT_ID = Reg.REGISTRANT_ID ' +
                                              ' AND eval.EVALUATION_FLAG = ''Y'') ';
 
  IF @EvaluatedLocationType = 'VISN' 
     SET @Query = @Query + 'AND EXISTS(SELECT * FROM AHOBPR.REGISTRANT_VISIT eval ' +
                                               'JOIN AHOBPR.VIEW_FACILITY_LOOKUP inst ON inst.ID = eval.STD_INSTITUTION_ID ' +
                                              'WHERE eval.REGISTRANT_ID = Reg.REGISTRANT_ID ' +
                                                'AND inst.VisnID = ''' + @EvaluatedLocationKey + ''' ' +
                                                'AND eval.EVALUATION_FLAG = ''Y'') ' 

  IF @EvaluatedLocationType = 'Base' 
     SET @Query = @Query + 'AND EXISTS(SELECT * FROM AHOBPR.REGISTRANT_VISIT eval ' +
                                               'JOIN AHOBPR.VIEW_FACILITY_LOOKUP inst ON inst.ID = eval.STD_INSTITUTION_ID ' +
                                              'WHERE eval.REGISTRANT_ID = Reg.REGISTRANT_ID ' +
                                                'AND inst.BASE_ID = ' + @EvaluatedLocationKey + ' ' + 
                                                'AND eval.EVALUATION_FLAG = ''Y'' ) '; 

  IF @EvaluatedLocationType = 'Institution' 
     SET @Query = @Query + 'AND EXISTS(SELECT * FROM AHOBPR.REGISTRANT_VISIT eval ' +
                                              'WHERE eval.REGISTRANT_ID = Reg.REGISTRANT_ID ' +
                                                'AND eval.STD_INSTITUTION_ID = ' + @EvaluatedLocationKey + ' ' +
                                                'AND eval.EVALUATION_FLAG = ''Y'' ) '; 

  PRINT  @Query;
  
  EXECUTE (@Query);  

  DROP TABLE #state;
  
END

-- EXEC [AHOBPR].[SP_ReportStatusByLocation] 
-- EXEC [AHOBPR].[SP_ReportStatusByLocation] @state = 'MD,VA'
-- EXEC [AHOBPR].[SP_ReportStatusByLocation] @status = '6,7'

-- exec [AHOBPR].[SP_ReportStatusByLocation]
-- exec [AHOBPR].[SP_ReportStatusByLocation] @AssignedLocationType = 'NLA'
-- exec [AHOBPR].[SP_ReportStatusByLocation] @AssignedLocationType = 'VISN', @AssignedLocationKey = 'V16'
-- exec [AHOBPR].[SP_ReportStatusByLocation] @AssignedLocationType = 'Base', @AssignedLocationKey = '1000089'
-- exec [AHOBPR].[SP_ReportStatusByLocation] @AssignedLocationType = 'Institution', @AssignedLocationKey = '1000089'
-- exec [AHOBPR].[SP_ReportStatusByLocation]
-- exec [AHOBPR].[SP_ReportStatusByLocation] @EvaluatedLocationType = 'NLA'
-- exec [AHOBPR].[SP_ReportStatusByLocation] @EvaluatedLocationType = 'VISN', @EvaluatedLocationKey = 'V16'
-- exec [AHOBPR].[SP_ReportStatusByLocation] @EvaluatedLocationType = 'Base', @EvaluatedLocationKey = '1000089'
-- exec [AHOBPR].[SP_ReportStatusByLocation] @EvaluatedLocationType = 'Institution', @EvaluatedLocationKey = '1000089'
GO

PRINT 'ALTER PROCEDURE SP_ReportRegistrantContact...';
GO

-- =============================================
-- Author:		Bruce Clegg
-- Create date: Jan 12, 2015
-- Description:	Used for the AHOBPR Contact Report  -- exec [AHOBPR].[SP_ReportRegistrantContact]    (More options at bottom)
-- 2015-02-23 - Clegg - provide individual field names
-- 20150330 - Bruce Clegg - Added VisnId value of NLA to find Null locations
-- 20150427 - Bruce Clegg - Change to LocationType and Key, for both Assigned and Evaluated Locations
-- 20150429 - Bruce Clegg - Added Email column (again)
-- 20150617 - D. Barr - Added state parameter
-- 20150722 - Bruce Clegg - Fix bug with @EvaluatedLocationType = 'Base'
-- 20150901 - Bruce Clegg Fix Main phone.  TYPE is not NULL so old way does not work
-- =============================================
ALTER PROCEDURE [AHOBPR].[SP_ReportRegistrantContact]
       @RegistrantStatus      VARCHAR(50) = '',    -- '1,2,3,4,5,6,7,8'
       @SvcStatus             VARCHAR(50) = '',    -- '1,2,3'
       @AdminFlagId           INT = 0,             -- 60
       @AdminFromDateText     VARCHAR(50) = NULL,
       @AdminToDateText       VARCHAR(50) = NULL,
       @SubPopulationFlagId   INT = 0,             -- 65
       @AssignedLocationType  VARCHAR(20) = NULL,  -- 'VISN', 'Base', 'Institution', 'NLA' = No Location Assigned
       @AssignedLocationKey   VARCHAR(50) = NULL,  -- 'V16', 'V01'    1000109,  1000089
       @EvaluatedLocationType VARCHAR(20) = NULL,  -- 'VISN', 'Base', 'Institution', 'NLA' = No Location Assigned
       @EvaluatedLocationKey  VARCHAR(50) = NULL,  -- 'V16', 'V01'    1000109,  1000089
       @CompletedFromDateText VARCHAR(50) = NULL,
       @CompletedToDateText   VARCHAR(50) = NULL,
       @state varchar(8000) = NULL -- Multiple states, comma-separated values when not NULL 
AS
BEGIN

DECLARE @AdminFromDate DATETIME2(0) = ISNULL(dbo.CleanDate(@AdminFromDateText),'19000101')
DECLARE @AdminToDate   DATETIME2(0) = dbo.AddMaxTime(ISNULL(dbo.CleanDate(@AdminToDateText),GETDATE())); 
DECLARE @CompletedFromDate DATETIME2(0) = ISNULL(dbo.CleanDate(@CompletedFromDateText),'19000101')
DECLARE @CompletedToDate   DATETIME2(0) = dbo.AddMaxTime(ISNULL(dbo.CleanDate(@CompletedToDateText),GETDATE())); 

--PRINT @AssignedLocationType;
--PRINT @AssignedLocationKey;

DECLARE @AssignedInstitutionId INT = 0,
        @EvaluatedInstitutionId INT = 0;

IF @AssignedLocationType IN ('Base','Institution') 
AND ISNUMERIC(@AssignedLocationKey) = 1
  SET @AssignedInstitutionId = CONVERT(INT,@AssignedLocationKey);
  
IF @EvaluatedLocationType IN ('Base','Institution') 
AND ISNUMERIC(@EvaluatedLocationKey) = 1
  SET @EvaluatedInstitutionId = CONVERT(INT,@EvaluatedLocationKey);

WITH addr 
AS (     
    SELECT REGISTRANT_ID,
		       ISNULL(STREET1,'')  AS STREET1, 
		       ISNULL(STREET2,'')  AS STREET2,
		       ISNULL(CITY,'')     AS CITY,
		       ISNULL([STATE],'')  AS [STATE],
		       ISNULL(COUNTRY,'')  AS COUNTRY,
		       ISNULL(ZIP,'')      AS ZIP
		  FROM  AHOBPR.[REGISTRANT_ADDRESS]
    )  
,phone1 
AS (     
    SELECT REGISTRANT_ID,		     
		       ISNULL(PHONE_NUMBER,'') AS PHONE_NUMBER_Main
      FROM AHOBPR.REGISTRANT_PHONE	
     WHERE ISNULL(TYPE,'') != 'Mobile'
    )       
,phone2 
AS (     
    SELECT REGISTRANT_ID,		     
		       ISNULL(PHONE_NUMBER,'') AS PHONE_NUMBER_Mobile
      FROM AHOBPR.REGISTRANT_PHONE	
     WHERE TYPE = 'Mobile'
    )
,email 
AS (     
    SELECT REGISTRANT_ID,		     
		       ISNULL(EMAIL,'') AS EMAIL
      FROM AHOBPR.REGISTRANT_EMAIL	
    )
,Pref    
AS(
      SELECT fr.REGISTRANT_ID 
           ,ans.TEXT AS ContactPreference
        FROM  AHOBPR.FORM_RESPONSE fr
        JOIN AHOBPR.FORM_RESPONSE_QUESTION rQst ON fr.FORM_RESPONSE_ID = rQst.FORM_RESPONSE_ID
        JOIN AHOBPR.STD_FORM_QUESTION qst ON rQst.STD_FORM_QUESTION_ID = qst.STD_FORM_QUESTION_ID
        JOIN AHOBPR.FORM_RESPONSE_ANSWER rAns ON rQst.FORM_RESPONSE_QUESTION_ID = rAns.FORM_RESPONSE_QUESTION_ID
        JOIN AHOBPR.STD_FORM_ANSWER ans ON ans.STD_FORM_ANSWER_ID = rAns.STD_FORM_ANSWER_ID
       WHERE qst.STD_FORM_ID = fr.STD_FORM_ID
         AND qst.FORM_QUESTION_UNIQUE_ID = '8.A'
 )           
  	SELECT 
  	      reg.REGISTRANT_ID,
		      ISNULL(reg.LAST_NAME,'') AS  LAST_NAME,
          ISNULL(reg.FIRST_NAME,'') AS  FIRST_NAME, 
          ISNULL(Pref.ContactPreference,'') AS  ContactPreference,
		      ISNULL(STREET1,'') AS  STREET1, 
		      ISNULL(STREET2,'') AS  STREET2,
		      ISNULL(CITY,'') AS  CITY,
		      ISNULL([STATE],'') AS  [STATE],
		      ISNULL(COUNTRY,'') AS  COUNTRY,
		      ISNULL(ZIP,'') AS  ZIP,
		      ISNULL(phone1.PHONE_NUMBER_Main,'') AS  PHONE_NUMBER_Main,
		      ISNULL(phone2.PHONE_NUMBER_Mobile,'') AS  PHONE_NUMBER_Mobile,
          ISNULL(email.EMAIL,'') AS EMAIL	
	  FROM AHOBPR.REGISTRANT reg	
	  LEFT JOIN addr  
		  ON reg.REGISTRANT_ID = addr.REGISTRANT_ID	 
	  LEFT JOIN phone1  
		  ON reg.REGISTRANT_ID = phone1.REGISTRANT_ID	 		  
	  LEFT JOIN phone2  
		  ON reg.REGISTRANT_ID = phone2.REGISTRANT_ID	 	
	  LEFT JOIN email  
		  ON reg.REGISTRANT_ID = email.REGISTRANT_ID	 	      		
	  LEFT JOIN Pref  
		  ON reg.REGISTRANT_ID = Pref.REGISTRANT_ID	 				
 	 WHERE (reg.STD_REGISTRANT_STATUS_ID IN (SELECT * FROM fncSplit(@RegistrantStatus, ',')) 
	        OR ISNULL(@RegistrantStatus,'') = '')
	   AND (reg.STD_SERVICE_STATUS_ID IN (SELECT * FROM fncSplit(@SvcStatus, ',')) 
	        OR ISNULL(@SvcStatus,'') = '')
     AND (@AssignedLocationType IS NULL
          OR (@AssignedLocationType = 'NLA' 
              AND reg.STD_INSTITUTION_ID IS NULL)
          OR (@AssignedLocationType = 'VISN' 
               AND EXISTS(SELECT * FROM AHOBPR.VIEW_FACILITY_LOOKUP inst
                                  WHERE inst.ID = reg.STD_INSTITUTION_ID
                                    AND inst.VisnID = @AssignedLocationKey)
             )	
          OR (@AssignedLocationType = 'Base' 
               AND EXISTS(SELECT * FROM AHOBPR.VIEW_FACILITY_LOOKUP inst
                                  WHERE inst.ID = reg.STD_INSTITUTION_ID
                                    AND inst.BASE_ID = @AssignedInstitutionId)
             )	
          OR (@AssignedLocationType = 'Institution' 
               AND reg.STD_INSTITUTION_ID = @AssignedInstitutionId)	
          )
     AND (@EvaluatedLocationType IS NULL
          OR (@EvaluatedLocationType = 'NLA' 
              AND NOT EXISTS( SELECT * FROM AHOBPR.REGISTRANT_VISIT eval
                               WHERE eval.REGISTRANT_ID = Reg.REGISTRANT_ID 
                                 AND eval.EVALUATION_FLAG = 'Y') 
             )
          OR (@EvaluatedLocationType = 'VISN' 
               AND EXISTS(SELECT * FROM AHOBPR.REGISTRANT_VISIT eval
                                   JOIN AHOBPR.VIEW_FACILITY_LOOKUP inst ON inst.ID = eval.STD_INSTITUTION_ID
                                  WHERE eval.REGISTRANT_ID = Reg.REGISTRANT_ID
                                    AND inst.VisnID = @EvaluatedLocationKey
                                    AND eval.EVALUATION_FLAG = 'Y') 
             )	
          OR (@EvaluatedLocationType = 'Base' 
               AND EXISTS(SELECT * FROM AHOBPR.REGISTRANT_VISIT eval
                                   JOIN AHOBPR.VIEW_FACILITY_LOOKUP inst ON inst.ID = eval.STD_INSTITUTION_ID
                                  WHERE eval.REGISTRANT_ID = Reg.REGISTRANT_ID
                                    AND inst.BASE_ID = @EvaluatedInstitutionId
                                    AND eval.EVALUATION_FLAG = 'Y' ) 	
             )	
          OR (@EvaluatedLocationType = 'Institution' 
               AND EXISTS(SELECT * FROM AHOBPR.REGISTRANT_VISIT eval
                                  WHERE eval.REGISTRANT_ID = Reg.REGISTRANT_ID
                                    AND eval.STD_INSTITUTION_ID = @EvaluatedInstitutionId
                                    AND eval.EVALUATION_FLAG = 'Y') 
             )	
          )          
     AND (ISNULL(@SubPopulationFlagId, 0) = 0 
          OR EXISTS(SELECT [REGISTRANT_ID] FROM AHOBPR.REGISTRANT_REGISTRY_FLAG popFlag
                     WHERE popFlag.REGISTRANT_ID = reg.REGISTRANT_ID
                       AND popFlag.STD_REGISTRY_FLAG_ID = @SubPopulationFlagId))
     AND (ISNULL(@AdminFlagId, 0) = 0 
          OR EXISTS(SELECT [REGISTRANT_ID] FROM AHOBPR.REGISTRANT_REGISTRY_FLAG adminFlag
                     WHERE adminFlag.REGISTRANT_ID = reg.REGISTRANT_ID
                       AND adminFlag.STD_REGISTRY_FLAG_ID = @AdminFlagId
                       AND adminFlag.CREATED BETWEEN @AdminFromDate AND @AdminToDate))   
     AND ((ISNULL(@CompletedFromDateText,'') = '' AND ISNULL(@CompletedToDateText,'') = '' )
         OR EXISTS(SELECT resp.REGISTRANT_ID, stat.QUESTIONNAIRE_COMPLETED_DATE
                     FROM AHOBPR.FORM_RESPONSE_STATUS stat
                     JOIN AHOBPR.FORM_RESPONSE resp
                       ON resp.FORM_RESPONSE_ID = stat.FORM_RESPONSE_ID
                    WHERE resp.REGISTRANT_ID = reg.REGISTRANT_ID
                      AND stat.QUESTIONNAIRE_COMPLETED_DATE BETWEEN @CompletedFromDate AND @CompletedToDate))
    AND (ISNULL(@state,'') = ''
	       OR EXISTS(SELECT DISTINCT addr.REGISTRANT_ID 
	                   FROM AHOBPR.REGISTRANT_ADDRESS addr 
	                  WHERE addr.REGISTRANT_ID = reg.REGISTRANT_ID 
	                    AND addr.STATE  IN (SELECT * FROM fncSplit(@state, ','))))
      ;
                                                                                                             
END
-- exec [AHOBPR].[SP_ReportRegistrantContact]
-- exec [AHOBPR].[SP_ReportRegistrantContact] @AssignedLocationType = 'NLA'
-- exec [AHOBPR].[SP_ReportRegistrantContact] @AssignedLocationType = 'VISN', @AssignedLocationKey = 'V16'
-- exec [AHOBPR].[SP_ReportRegistrantContact] @AssignedLocationType = 'Base', @AssignedLocationKey = '1000089'
-- exec [AHOBPR].[SP_ReportRegistrantContact] @AssignedLocationType = 'Institution', @AssignedLocationKey = '1000089'
-- exec [AHOBPR].[SP_ReportRegistrantContact]
-- exec [AHOBPR].[SP_ReportRegistrantContact] @EvaluatedLocationType = 'NLA'
-- exec [AHOBPR].[SP_ReportRegistrantContact] @EvaluatedLocationType = 'VISN', @EvaluatedLocationKey = 'V16'
-- exec [AHOBPR].[SP_ReportRegistrantContact] @EvaluatedLocationType = 'Base', @EvaluatedLocationKey = '1000089'
-- exec [AHOBPR].[SP_ReportRegistrantContact] @EvaluatedLocationType = 'Institution', @EvaluatedLocationKey = '1000089'


       --@AssignedLocationType  VARCHAR(10) = NULL,  -- 'VISN', 'Base', 'Institution', 'NLA' = No Location Assigned
       --@AssignedLocationKey   VARCHAR(50) = NULL,  -- 'V16', 'V01'    1000109,  1000089
 --exec [AHOBPR].[SP_ReportRegistrantContact] '5,6,7', NULL, 0, NULL, NULL, 0, 0, NULL  @EvaluatedLocationType
 --exec [AHOBPR].[SP_ReportRegistrantContact] NULL, '2', 0, NULL, NULL, 0, 0, NULL
 --exec [AHOBPR].[SP_ReportRegistrantContact] NULL, NULL, 60, NULL, '20141204', 0, 0, NULL
 --exec [AHOBPR].[SP_ReportRegistrantContact] NULL, NULL, 60, '12/04/2014', NULL, 0, 0, NULL
 --exec [AHOBPR].[SP_ReportRegistrantContact] NULL, NULL, 60, '12/03/2014', '12/12/2014', 0, 0, NULL
 --exec [AHOBPR].[SP_ReportRegistrantContact] NULL, NULL, 0, NULL, NULL, 65, 0, NULL
 --exec [AHOBPR].[SP_ReportRegistrantContact] NULL, NULL, 0, NULL, NULL, 0, 1000109, NULL
 --exec [AHOBPR].[SP_ReportRegistrantContact] NULL, NULL, 0, NULL, NULL, 0, 0, 'NLA' --'V01'
 --exec [AHOBPR].[SP_ReportRegistrantContact] @CompletedFromDateText = '08/13/2014'
 --exec [AHOBPR].[SP_ReportRegistrantContact] @CompletedToDateText = '08/13/2014'
 --exec [AHOBPR].[SP_ReportRegistrantContact] @CompletedFromDateText = '08/13/2014', @CompletedToDateText = '08/13/2014'
GO

PRINT 'ALTER PROCEDURE SP_ReportAdHocByLocation...';
GO
-- =============================================
-- Author:		Deborah Barr
-- Create date: November 12, 2014
-- Description:	Used for the AHOBPR Ad Hoc Report  

-- Modification log
-- 2014-12-31 - Changed from FLAG to RESPONSE for Health Concerns and REQUEST_SEEN, Reformatted statements for readability
-- 2015-01-05 - Change Date Format to MM/DD/YYYY
-- 2015-01-14 - Sherry Wang Add more parameters: @healthFactor, @formStartFromDate, @formStartToDate, @subpopulationFlag
-- 2015-01-14 - Sherry Wang Add more parameters: @adminFlag, @deploymentAll, @deploymentExposure, @adminFlagFromDate, @adminFlagToDate
-- 2015-01-16 - Bruce Clegg - Fix parameters, Add distinct and outer query to group counts.
-- 2015-02-09 - Bruce Clegg - Change Health Factor Exists statements to use new REGISTRANT_VISIT
-- 2015-05-07 - Bruce Clegg - Change Facility name from "Other" to "No Facitilty Assigned" when not found
-- 2015-06-04 - Bruce Clegg - Add more parameters
-- 2015-06-18 - Bruce Clegg - Changed @separationType to an Integer 
-- 2015-07-09 - Bruce Clegg - Rewrote as Dynamic SQL for performance
-- 2015-08-05 - Bruce Clegg - Fix bug with AdminFlag - Set From and To date defaults
-- =============================================
ALTER PROCEDURE [AHOBPR].[SP_ReportAdHocByLocation]
       @status varchar(100) = NULL, -- Multiple statues, comma-separated values when not NULL
       @dutyStatus varchar(100) = NULL, -- Multiple states, comma-separated values when not NULL  
       @formStartFromDateText varchar(100) = NULL,
       @formStartToDateText varchar(100) = NULL,
       @healthFactor varchar(100) = NULL, -- Multiple health factor IDs, comma-separated values when not NULL
       @formCompletionFromDateText varchar(100) = NULL,
       @formCompletionToDateText varchar(100) = NULL,
       @questionNumber varchar(100) = NULL,
       @answer varchar(100) = NULL,
       @healthConcerns varchar(10) = 'All', 
       @requestToBeSeen varchar(10) = 'All',
       @groupBy VARCHAR(50) = '',                 -- 'VISN' or 'Facility'   
       @subpopulationFlag varchar (10) = NULL, 
       @adminFlag varchar (10) = NULL, 
       @deploymentAll varchar (256) = NULL, 
       @deploymentExposure varchar (256) = NULL, 
       @adminFlagFromDateText varchar (20) = NULL, 
       @adminFlagToDateText varchar (20) = NULL,
       @excludeDead varchar(1) = 'N',
       @gender varchar(50) = '',
       @race varchar(50) = '',
       @ethnicity varchar(50) = '',
       @separationType INT = 0,
       @separationFromDateText varchar (20) = NULL, 
       @separationToDateText varchar (20) = NULL, 
       @enrollmentStatus varchar(50) = ''
AS

BEGIN

  DECLARE @formStartFromDate DATETIME2(0) = NULL,
          @formStartToDate DATETIME2(0) = NULL,
          @formCompletionFromDate DATETIME2(0) = NULL,
          @formCompletionToDate DATETIME2(0) = NULL,
          @adminFlagFromDate DATETIME2(0) = NULL,
          @adminFlagToDate DATETIME2(0) = NULL,
          @separationFromDate DATETIME2(0) = NULL,
          @separationToDate DATETIME2(0) = NULL;
          
  DECLARE @Query VARCHAR(8000);      

  -- Convert date parameters      

  SET @formStartFromDate = dbo.CleanDate(@formStartFromDateText);
  SET @formStartToDate = dbo.AddMaxTime(dbo.CleanDate(@formStartToDateText));
  SET @formCompletionFromDate = dbo.CleanDate(@formCompletionFromDateText);
  SET @formCompletionToDate = dbo.AddMaxTime(dbo.CleanDate(@formCompletionToDateText));     
  SET @adminFlagFromDate = ISNULL(dbo.CleanDate(@adminFlagFromDateText),'19000101');
  SET @adminFlagToDate = dbo.AddMaxTime(ISNULL(dbo.CleanDate(@adminFlagToDateText),GETDATE())); 
  SET @separationFromDate = dbo.CleanDate(@separationFromDateText);
  SET @separationToDate = dbo.AddMaxTime(dbo.CleanDate(@separationToDateText));   

  -- Build Src TDE
  SET @Query = ' WITH Src 
    AS( SELECT a.REGISTRANT_ID, a.STD_INSTITUTION_ID FROM AHOBPR.REGISTRANT a WHERE 1 = 1 ';
  
  IF @status IS NOT NULL
    SET @Query = @Query + ' AND a.STD_REGISTRANT_STATUS_ID IN (' + @status + ')';

  IF ISNULL(@dutyStatus,'') > ''
    SET @Query = @Query + ' AND a.STD_SERVICE_STATUS_ID IN (' + @dutyStatus + ')';
          
  IF @healthConcerns != 'All'
    SET @Query = @Query + ' AND a.HEALTH_CONCERNS_RESPONSE = ''' + @healthConcerns + ''''; 
  
  IF @requestToBeSeen != 'All'
    SET @Query = @Query + ' AND a.REQUEST_SEEN_RESPONSE = ''' + @requestToBeSeen  + '''';
    
  IF @formStartFromDate IS NOT NULL OR @formStartToDate IS NOT NULL 
  OR @formCompletionFromDate IS NOT NULL OR @formCompletionToDate IS NOT NULL
    BEGIN
      SET @Query = @Query + ' AND EXISTS( SELECT d.REGISTRANT_ID FROM AHOBPR.FORM_RESPONSE d 
		                        JOIN AHOBPR.FORM_RESPONSE_STATUS e ON d.FORM_RESPONSE_ID = e.FORM_RESPONSE_ID	
		                       WHERE a.REGISTRANT_ID = d.REGISTRANT_ID ';
		  IF @formStartFromDate IS NOT NULL                     
		     SET @Query = @Query + ' AND e.QUESTIONNAIRE_STARTED_DATE >= ''' + CONVERT(VARCHAR(20),@formStartFromDate,121) + '''';
		  IF @formStartToDate IS NOT NULL                     
		     SET @Query = @Query + ' AND e.QUESTIONNAIRE_STARTED_DATE <= ''' + CONVERT(VARCHAR(20),@formStartToDate,121) + '''';
		  IF @formCompletionFromDate IS NOT NULL                     
		     SET @Query = @Query + ' AND e.QUESTIONNAIRE_COMPLETED_DATE >= ''' + CONVERT(VARCHAR(20),@formCompletionFromDate,121) + '''';
		  IF @formCompletionToDate IS NOT NULL                     
		     SET @Query = @Query + ' AND e.QUESTIONNAIRE_COMPLETED_DATE <= ''' + CONVERT(VARCHAR(20),@formCompletionToDate,121) + '''';
		   SET @Query = @Query + ' ) ' -- Close Exisits statement
    END    
    
  IF ISNULL(@answer,'') != '' 
    --SET @Query = @Query + ' AND EXISTS(SELECT fr.REGISTRANT_ID FROM AHOBPR.FORM_RESPONSE_ANSWER ra ' +
    --                      'JOIN AHOBPR.FORM_RESPONSE_QUESTION rq ON rq.FORM_RESPONSE_QUESTION_ID = ra.FORM_RESPONSE_QUESTION_ID ' +
    --                      'JOIN AHOBPR.FORM_RESPONSE fr ON fr.FORM_RESPONSE_ID = rq.FORM_RESPONSE_ID ' +
    --                    ' WHERE ra.STD_FORM_ANSWER_ID IN (' + @answer + ') AND fr.REGISTRANT_ID = a.REGISTRANT_ID )';
                        
  SET @Query = @Query + ' AND EXISTS(SELECT fr.REGISTRANT_ID FROM AHOBPR.FORM_RESPONSE_ANSWER ra ' + 
                        'JOIN AHOBPR.FORM_RESPONSE_QUESTION rq ON rq.FORM_RESPONSE_QUESTION_ID = ra.FORM_RESPONSE_QUESTION_ID  ' + 
                        'JOIN AHOBPR.FORM_RESPONSE fr ON fr.FORM_RESPONSE_ID = rq.FORM_RESPONSE_ID  ' + 
                        'JOIN AHOBPR.STD_FORM_ANSWER A1 ON A1.STD_FORM_ANSWER_ID = ra.STD_FORM_ANSWER_ID ' + 
                        'JOIN AHOBPR.STD_FORM_QUESTION Q1 ON Q1.STD_FORM_QUESTION_ID = rq.STD_FORM_QUESTION_ID ' + 
                        'JOIN AHOBPR.STD_FORM_QUESTION Q2 ON Q2.FORM_QUESTION_UNIQUE_ID = Q1.FORM_QUESTION_UNIQUE_ID ' + 
                        'JOIN AHOBPR.STD_FORM_ANSWER A2 ON A2.STD_FORM_QUESTION_ID = Q2.STD_FORM_QUESTION_ID ' + 
                        ' AND A1.BRP_FORM_ANSWER_ID = A2.BRP_FORM_ANSWER_ID ' + 
                      ' WHERE A2.STD_FORM_ANSWER_ID IN (' + @answer + ') AND fr.REGISTRANT_ID = a.REGISTRANT_ID )';                         
                        
  
  IF ISNULL(@subpopulationFlag,'') != ''
    SET @Query = @Query + ' AND EXISTS(SELECT * FROM AHOBPR.REGISTRANT_REGISTRY_FLAG subpop ' +
                                      ' WHERE subpop.STD_REGISTRY_FLAG_ID IN (' + @subpopulationFlag + ') ' +
                                        ' AND subpop.REGISTRANT_ID = a.REGISTRANT_ID ) ';

	IF ISNULL(@adminFlag,'') != ''
    SET @Query = @Query + ' AND EXISTS(SELECT * FROM AHOBPR.REGISTRANT_REGISTRY_FLAG adminflag ' +
                        ' WHERE adminflag.STD_REGISTRY_FLAG_ID IN (' + @adminFlag + ') ' +
                          ' AND adminflag.REGISTRANT_ID = a.REGISTRANT_ID ' +
                          ' AND adminflag.CREATED BETWEEN ''' + CONVERT(VARCHAR(20),@adminFlagFromDate,121) + 
                                                  ''' AND ''' + CONVERT(VARCHAR(20),@adminFlagToDate,121) + ''' ) ';  
                            
  IF ISNULL(@healthFactor,'') != ''
    SET @Query = @Query + ' AND EXISTS (SELECT re.REGISTRANT_ID FROM AHOBPR.REGISTRANT_VISIT re ' +
                                        ' JOIN AHOBPR.REGISTRANT_HEALTH_FACTOR hf ON re.REGISTRANT_VISIT_ID = hf.REGISTRANT_VISIT_ID ' +
                                       ' WHERE a.REGISTRANT_ID = re.REGISTRANT_ID AND re.EVALUATION_FLAG = ''Y'' ' +
                                       '   AND hf.STD_HEALTH_FACTOR_ID IN (' + @healthFactor + ') ) '

  IF ISNULL(@deploymentAll,'') != '' 
    SET @Query = @Query + ' AND EXISTS (SELECT depl.REGISTRANT_ID FROM AHOBPR.REGISTRANT_DEPLOYMENT depl ' +
                                       ' WHERE a.REGISTRANT_ID = depl.REGISTRANT_ID
                                           AND depl.DEPLOYMENT_COUNTRY = ''' + @deploymentAll + ''') ' ;

  IF ISNULL(@deploymentExposure,'') != '' 
    SET @Query = @Query + ' AND EXISTS(SELECT deploymentExposure.REGISTRANT_ID FROM AHOBPR.REGISTRANT_DEPLOYMENT deploymentExposure ' +
                                        '  JOIN AHOBPR.FORM_RESPONSE_QUESTION exposure ON deploymentExposure.REGISTRANT_DEPLOYMENT_ID = exposure.REGISTRANT_DEPLOYMENT_ID ' +
                                        '  JOIN AHOBPR.STD_FORM_QUESTION quest ON exposure.STD_FORM_QUESTION_ID = quest.STD_FORM_QUESTION_ID ' +
                                        ' WHERE quest.FORM_QUESTION_UNIQUE_ID = ''1.2.A'' ' +
                                        '   AND deploymentExposure.DEPLOYMENT_COUNTRY = ''' + @deploymentExposure + '''' + 
                                        '   AND deploymentExposure.REGISTRANT_ID = a.REGISTRANT_ID ) ';
  IF ISNULL(@excludeDead,'N') = 'Y'
    SET @Query = @Query + ' AND a.DECEASED_FLAG = 0 ';
  
  IF ISNULL(@gender,'') != ''
    SET @Query = @Query + ' AND ''' + @gender + ''' = ISNULL(a.GENDER,''No Data Available'') ';
    
  IF ISNULL(@ethnicity,'') != ''
    SET @Query = @Query + ' AND ''' + @ethnicity + ''' = ISNULL(a.ETHNICITY,''No Data Available'') ';
  
  IF ISNULL(@separationType,0) != 0
    SET @Query = @Query + ' AND ''' + @separationType + ''' = ISNULL(a.STD_SEPARATION_REASON_ID_CURRENT,0) ';

  IF @separationFromDate IS NOT NULL 
    SET @Query = @Query + ' AND a.SEPARATION_DATE_CURRENT >= ''' + CONVERT(VARCHAR(20),@separationFromDate,121) + '''';
  
  IF @separationToDate IS NOT NULL 
    SET @Query = @Query + ' AND  a.SEPARATION_DATE_CURRENT <= ''' + CONVERT(VARCHAR(20),@separationToDate,121) + '''';
  
  IF ISNULL(@race,'') != ''
    SET @Query = @Query + ' AND ''' + @race + ''' =  ISNULL(a.RACE ,''No Data Available'') ';
    
  IF ISNULL(@enrollmentStatus,'') != ''
    SET @Query = @Query + ' AND ''' + @enrollmentStatus + ''' = ISNULL(a.ENROLLMENT_STATUS,''No Data Available'') ';

  -- Add Summary TDE                   
  SET @Query = @Query + '  )	,Summary  AS( ' + 
                        ' SELECT Src.STD_INSTITUTION_ID, COUNT(*) AS TotalRegistrants FROM Src ' +
                        ' GROUP BY STD_INSTITUTION_ID ) ';
-- By Facility 
	IF @groupBy = 'Facility'
	  SET @Query = @Query + ' SELECT inst.Name AS Location, ' +
	                                ' ISNULL(CONVERT(VARCHAR(10),Summary.STD_INSTITUTION_ID),''-1'') AS LocationId, ' +
                                   '  SUM(TotalRegistrants) AS TotalRegistrants FROM Summary ' +		       
	                         '  JOIN AHOBPR.VIEW_INSTITUTION_NAME inst ON inst.ID = ISNULL(Summary.STD_INSTITUTION_ID,-1) ' +
	                         ' GROUP BY inst.Name,STD_INSTITUTION_ID ' +
  		                     ' OPTION (OPTIMIZE FOR UNKNOWN);';
  
-- By VISN
	IF @groupBy = 'VISN'
	   SET @Query = @Query + ' ,VisnSummary AS ( ' +
                 ' SELECT ISNULL(inst.VisnID,''NLA'') AS Location, ' +
                       '  ISNULL(inst.VisnID, ''-1'') AS LocationId, ' +
                       '  SUM(TotalRegistrants) AS TotalRegistrants ' +		       
                 '   FROM Summary ' +   
                 '   LEFT JOIN AHOBPR.VIEW_FACILITY_LOOKUP inst ON inst.ID = ISNULL(Summary.STD_INSTITUTION_ID,-1) ' +  
                 '   GROUP BY inst.VisnID  ) ' +
                 ' SELECT REPLACE(Location,''NLA'',''No Facitilty Assigned'') AS Location, ' +
                 '        CONVERT(VARCHAR(10),LocationId) AS LocationId, TotalRegistrants ' +
                 '   FROM VisnSummary  ' +
                 '  ORDER BY Location ' + 
                 '  OPTION (OPTIMIZE FOR UNKNOWN); ' ;
	
  PRINT @Query;		
  		                     
  EXECUTE (@Query);  		                     		
END
-- Parameters
  --@status, @dutyStatus ,@questionnaireStartDate, @questionnaireCompleteDate, 
  --@questionNumber, @answer, @healthConcerns= 'All', @requestToBeSeen = 'All',
  --@registryFlag,@InstitutionId = 0, @VisnId = '' 
-- Test Cases
-- EXECUTE [AHOBPR].[SP_ReportAdHocByLocation] @groupBy = 'VISN',  @status = '6,7'
-- EXECUTE [AHOBPR].[SP_ReportAdHocByLocation] @groupBy = 'VISN',  @dutyStatus = '1,2'
-- EXECUTE [AHOBPR].[SP_ReportAdHocByLocation] @groupBy = 'VISN',  @formStartFromDateText  = '20010101'
-- EXECUTE [AHOBPR].[SP_ReportAdHocByLocation] @groupBy = 'VISN',  @formStartToDateText = '20141231'
-- EXECUTE [AHOBPR].[SP_ReportAdHocByLocation] @groupBy = 'VISN',  @healthFactor = '44,45,46,47' 
-- EXECUTE [AHOBPR].[SP_ReportAdHocByLocation] @groupBy = 'VISN',  @formCompletionFromDateText  = '20010101'
-- EXECUTE [AHOBPR].[SP_ReportAdHocByLocation] @groupBy = 'VISN',  @formCompletionToDateText  = '20141231'
-- EXECUTE [AHOBPR].[SP_ReportAdHocByLocation] @groupBy = 'VISN',  @healthConcerns = 'Yes' 
-- EXECUTE [AHOBPR].[SP_ReportAdHocByLocation] @groupBy = 'VISN',  @requestToBeSeen = 'Yes'
-- EXECUTE [AHOBPR].[SP_ReportAdHocByLocation] @groupBy = 'VISN',  @subpopulationFlag = '64,65,66,67'
-- EXECUTE [AHOBPR].[SP_ReportAdHocByLocation] @groupBy = 'VISN',  @deploymentAll = 'Iraq'    -- 'Egypt'
-- EXECUTE [AHOBPR].[SP_ReportAdHocByLocation] @groupBy = 'VISN',  @deploymentExposure  = 'Iraq'
-- EXECUTE [AHOBPR].[SP_ReportAdHocByLocation] @groupBy = 'VISN',  @adminFlag = '59,60' ,@adminFlagFromDateText = '20150101' , @adminFlagToDateText = '20150201'
-- EXECUTE [AHOBPR].[SP_ReportAdHocByLocation] @groupBy = 'VISN',  @answer = '2221'
-- EXECUTE [AHOBPR].[SP_ReportAdHocByLocation] @groupBy =  'Facility' , @status = '6,7' ,@answer = '2346,1571,17'   -- '20,795,1570' 
GO

PRINT 'ALTER PROCEDURE SP_ReportAdHocByIndividual...';
GO
-- =============================================
-- Author:		Deborah Barr
-- Create date: November 12, 2014
-- Description:	Used for the AHOBPR Ad Hoc Report  

-- Modification log
-- 2014-12-31 - Changed from FLAG to RESPONSE for Health Concerns and REQUEST_SEEN, Reformatted statements for readability
-- 2015-01-05 - Change Date Format to MM/DD/YYYY
-- 2015-01-14 - Sherry Wang Add more parameters: @healthFactor, @formStartFromDate, @formStartToDate, @subpopulationFlag
-- 2015-01-14 - Sherry Wang Add more parameters: @adminFlag, @deploymentAll, @deploymentExposure, @adminFlagFromDate, @adminFlagToDate
-- 2015-01-16 - Bruce Clegg - Fix parameters and remove NULLs from display where possible
-- 2015-02-09 - Bruce Clegg - Change Health Factor Exists statements to use new REGISTRANT_VISIT
-- 2015-03-30 - Bruce Clegg - Added VisnId value of NLA to find Null locations
-- 2015-04-27 - Bruce Clegg - Change to LocationType and Key, for both Assigned and Evaluated Locations
-- 2015-06-04 - Bruce Clegg - Add more parameters
-- 2015-06-18 - Bruce Clegg - Changed @separationType to an Integer 
-- 2015-07-09 - Bruce CLegg - Rewrote as Dynamic SQL for performance
-- =============================================
ALTER PROCEDURE [AHOBPR].[SP_ReportAdHocByIndividual]
       @status varchar(100) = NULL, -- Multiple statues, comma-separated values when not NULL
       @dutyStatus varchar(100) = NULL, -- Multiple states, comma-separated values when not NULL  
       @formStartFromDateText varchar(100) = NULL,
       @formStartToDateText varchar(100) = NULL,
       @healthFactor varchar(100) = NULL, -- Multiple health factor IDs, comma-separated values when not NULL
       @formCompletionFromDateText varchar(100) = NULL,
       @formCompletionToDateText varchar(100) = NULL,
       @questionNumber varchar(100) = NULL,
       @answer varchar(100) = NULL,
       @healthConcerns varchar(100) = 'All', 
       @requestToBeSeen varchar(100) = 'All',
       @AssignedLocationType  VARCHAR(20) = NULL,  -- 'VISN', 'Base', 'Institution', 'NLA' = No Location Assigned
       @AssignedLocationKey   VARCHAR(50) = NULL,  -- 'V16', 'V01'    1000109,  1000089
       @EvaluatedLocationType VARCHAR(20) = NULL,  -- 'VISN', 'Base', 'Institution', 'NLA' = No Location Assigned
       @EvaluatedLocationKey  VARCHAR(50) = NULL,  -- 'V16', 'V01'    1000109,  1000089
       @subpopulationFlag varchar (10) = NULL, 
       @adminFlag varchar (10) = NULL, 
       @deploymentAll varchar (256) = NULL, 
       @deploymentExposure varchar (256) = NULL, 
       @adminFlagFromDateText varchar (20) = NULL, 
       @adminFlagToDateText varchar (20) = NULL,
       @excludeDead varchar(1) = 'N',
       @gender varchar(50) = '',
       @race varchar(50) = '',
       @ethnicity varchar(50) = '',
       @separationType INT = 0,
       @separationFromDateText varchar (20) = NULL, 
       @separationToDateText varchar (20) = NULL, 
       @enrollmentStatus varchar(50) = ''
AS

BEGIN

-- Convert date parameters from VARCHAR to DATETIME2(0)    
  DECLARE @formStartFromDate DATETIME2(0) = NULL,
          @formStartToDate DATETIME2(0) = NULL,
          @formCompletionFromDate DATETIME2(0) = NULL,
          @formCompletionToDate DATETIME2(0) = NULL,
          @adminFlagFromDate DATETIME2(0) = NULL,
          @adminFlagToDate DATETIME2(0) = NULL,
          @separationFromDate DATETIME2(0) = NULL,
          @separationToDate DATETIME2(0) = NULL;

  DECLARE @Query VARCHAR(8000);                
  DECLARE @MainQuery VARCHAR(8000);                
  DECLARE @RegCriteria VARCHAR(8000) = '';                
  DECLARE @AdminWith VARCHAR(8000);                
  DECLARE @SubPopWith VARCHAR(8000);                
  DECLARE @DeployWith VARCHAR(8000);                

  SET @formStartFromDate = dbo.CleanDate(@formStartFromDateText);
  SET @formStartToDate = dbo.AddMaxTime(dbo.CleanDate(@formStartToDateText));
  SET @formCompletionFromDate = dbo.CleanDate(@formCompletionFromDateText);
  SET @formCompletionToDate = dbo.AddMaxTime(dbo.CleanDate(@formCompletionToDateText));     
  SET @adminFlagFromDate = ISNULL(dbo.CleanDate(@adminFlagFromDateText),'19000101');
  SET @adminFlagToDate = dbo.AddMaxTime(ISNULL(dbo.CleanDate(@adminFlagToDateText),GETDATE())); 
	SET @separationFromDate = dbo.CleanDate(@separationFromDateText);
  SET @separationToDate = dbo.AddMaxTime(dbo.CleanDate(@separationToDateText)); 

	DECLARE @AdminType INT = (SELECT STD_REGISTRY_FLAG_TYPE_ID
	                            FROM AHOBPR.STD_REGISTRY_FLAG_TYPE 
	                           WHERE REGISTRY_FLAG_TYPE = 'Administrative Flag');
	                           
	DECLARE @SubPopType INT = (SELECT STD_REGISTRY_FLAG_TYPE_ID
	                            FROM AHOBPR.STD_REGISTRY_FLAG_TYPE 
	                           WHERE REGISTRY_FLAG_TYPE = 'SubPopulation Flag');	                           
	                            	
-- Set InstitutionID values if needed.
DECLARE @AssignedInstitutionId INT = 0,
        @EvaluatedInstitutionId INT = 0;

IF @AssignedLocationType IN ('Base','Institution') 
AND ISNUMERIC(@AssignedLocationKey) = 1
  SET @AssignedInstitutionId = CONVERT(INT,@AssignedLocationKey);
  
IF @EvaluatedLocationType IN ('Base','Institution') 
AND ISNUMERIC(@EvaluatedLocationKey) = 1
  SET @EvaluatedInstitutionId = CONVERT(INT,@EvaluatedLocationKey);

  --Concatenate one or more Sub-population flags for Registrants with flags
  SET @SubPopWith = 'WITH SubPop AS ( SELECT a.REGISTRANT_ID, '+
                                             'STUFF((SELECT '', '' + c.REGISTRY_FLAG_NAME ' +
                                                     ' FROM AHOBPR.REGISTRANT_REGISTRY_FLAG b ' +
                                                     ' JOIN AHOBPR.STD_REGISTRY_FLAG c ON b.STD_REGISTRY_FLAG_ID = c.STD_REGISTRY_FLAG_ID ' +
                                                    ' WHERE a.REGISTRANT_ID = b.REGISTRANT_ID ' +
                                                    '   AND c.STD_REGISTRY_FLAG_TYPE_ID = ' + CONVERT(VARCHAR(3),@SubPopType) +
                                                      ' FOR XML PATH('''')), 1, 1, '''') AS [REGISTRY_FLAG_NAME] ' +
                                     ' FROM AHOBPR.REGISTRANT a ' +
                                     'WHERE 1 = 1 ';
                                     
  --Concatenate one or more Admin flags for Registrants with flags
  SET @AdminWith = ', Admin AS (  SELECT a.REGISTRANT_ID, ' +
                                 ' STUFF((SELECT '', '' + c.REGISTRY_FLAG_NAME ' +
                                         '  FROM AHOBPR.REGISTRANT_REGISTRY_FLAG b ' +
                                         '  JOIN AHOBPR.STD_REGISTRY_FLAG c ON b.STD_REGISTRY_FLAG_ID = c.STD_REGISTRY_FLAG_ID ' +
                                         ' WHERE a.REGISTRANT_ID = b.REGISTRANT_ID ' +
                                         '   AND c.STD_REGISTRY_FLAG_TYPE_ID = ' + CONVERT(VARCHAR(3),@AdminType) +
                                         '   AND b.CREATED BETWEEN ''' + CONVERT(VARCHAR(20),@adminFlagFromDate,121) + 
                                                           ''' AND ''' + CONVERT(VARCHAR(20),@adminFlagToDate,121) + '''' + 
                                         '   FOR XML PATH('''')), 1, 1, '''') AS [REGISTRY_FLAG_NAME] ' +
                               '  FROM AHOBPR.REGISTRANT a ' +
                              '  WHERE 1 = 1 ';

  --Concatenate one or more Deployment Countries for Registrants with Deployments
  SET @DeployWith = ', Deploy AS (SELECT a.REGISTRANT_ID,  ' +
                                       ' STUFF((SELECT '', '' + b.DEPLOYMENT_COUNTRY ' + 
                                               '  FROM AHOBPR.REGISTRANT_DEPLOYMENT b ' +
                                              '  WHERE a.REGISTRANT_ID = b.REGISTRANT_ID ' +
                                              '    FOR XML PATH('''')), 1, 1, '''') AS DEPLOYMENT_COUNTRY ' +
                                  ' FROM AHOBPR.REGISTRANT a ' +
                                '  WHERE 1 = 1 ';

  SET @MainQuery = '  SELECT DISTINCT a.REGISTRANT_ID ' +
                            ',(a.LAST_NAME + '', '' + a.FIRST_NAME) AS Name ' +
                            ',b.REGISTRANT_STATUS AS Status ' +
                            ',c.SERVICE_STATUS AS DutyStatus ' +
                            ',e.QUESTIONNAIRE_STARTED_DATE AS QuestionStartDate ' +
                            ',e.QUESTIONNAIRE_COMPLETED_DATE AS QuestionCompleteDate ' +
                            ',ISNULL(CONVERT(VARCHAR(10), e.QUESTIONNAIRE_STARTED_DATE, 101),'''') AS QuestionStartDateDisplay ' +
                            ',ISNULL(CONVERT(VARCHAR(10), e.QUESTIONNAIRE_COMPLETED_DATE, 101),'''') AS QuestionCompleteDateDisplay ';
--Decide which query to run depending on if a question was selected as a parameter for the report
  IF ISNULL(@questionNumber, '') = ''
    SET @MainQuery = @MainQuery + ','''' AS QuestionNumber ' +
                                  ','''' AS Answer '
  ELSE                                  
    SET @MainQuery = @MainQuery + ',''' + @questionNumber + ''' AS QuestionNumber ' +
                                  ',STUFF((SELECT '', '' + stdAnswer.TEXT ' +
                                          '  FROM AHOBPR.FORM_RESPONSE_ANSWER answer ' + 
                                          '  JOIN AHOBPR.STD_FORM_ANSWER stdAnswer ON answer.STD_FORM_ANSWER_ID = stdAnswer.STD_FORM_ANSWER_ID ' +
                                          ' WHERE formQuestion.FORM_RESPONSE_QUESTION_ID = answer.FORM_RESPONSE_QUESTION_ID ' +
                                          ' FOR XML PATH('''')), 1, 1, '''') AS Answer ' ;                  
    
  SET @MainQuery = @MainQuery + ',ISNULL(a.HEALTH_CONCERNS_RESPONSE,'''') AS HealthConcerns ' +
                                ',ISNULL(a.REQUEST_SEEN_RESPONSE,'''')  AS RequestBeSeen  ' +
                                ',RTRIM(LTRIM(ISNULL(SubPop.REGISTRY_FLAG_NAME,''''))) AS SubpopulationFlag  ' +
                                ',RTRIM(LTRIM(ISNULL(Admin.REGISTRY_FLAG_NAME,''''))) AS AdminFlag  ' +
                                ',ISNULL(Deploy.DEPLOYMENT_COUNTRY,'''') AS DeploymentCountry		 ' +
		                            ',a.BIRTH_DATE AS BirthDate  ' +
		                            ',CONVERT(VARCHAR(10), a.BIRTH_DATE, 101) AS BirthDateDisplay ' +
	                        'FROM AHOBPR.REGISTRANT a '
	                        
  SET @MainQuery = @MainQuery + ' JOIN AHOBPR.STD_REGISTRANT_STATUS b ON a.STD_REGISTRANT_STATUS_ID = b.STD_REGISTRANT_STATUS_ID ' +
	                              ' JOIN AHOBPR.STD_SERVICE_STATUS c ON a.STD_SERVICE_STATUS_ID = c.STD_SERVICE_STATUS_ID ' +
		                            ' LEFT JOIN AHOBPR.FORM_RESPONSE d ON a.REGISTRANT_ID =  d.REGISTRANT_ID ' +
		                            ' LEFT JOIN AHOBPR.FORM_RESPONSE_STATUS e ON d.FORM_RESPONSE_ID = e.FORM_RESPONSE_ID ' +
		                            ' LEFT JOIN SubPop  ON a.REGISTRANT_ID = SubPop.REGISTRANT_ID ' +
		                            ' LEFT JOIN Admin   ON a.REGISTRANT_ID = Admin.REGISTRANT_ID ' +
		                            ' LEFT JOIN Deploy  ON a.REGISTRANT_ID = Deploy.REGISTRANT_ID ';

  IF ISNULL(@questionNumber, '') != ''
    SET @MainQuery = @MainQuery + ' JOIN AHOBPR.FORM_RESPONSE_QUESTION formQuestion ON d.FORM_RESPONSE_ID = formQuestion.FORM_RESPONSE_ID ' +
                                  ' JOIN AHOBPR.STD_FORM_QUESTION stdQuestion ON formQuestion.STD_FORM_QUESTION_ID = stdQuestion.STD_FORM_QUESTION_ID ';

  SET @MainQuery = @MainQuery + ' WHERE 1 = 1 ';

  -- Question Number and Answer criteria are only used against Main Query		  
  IF ISNULL(@questionNumber, '') > ''
	  SET @MainQuery = @MainQuery + ' AND stdQuestion.FORM_QUESTION_UNIQUE_ID = ''' + @questionNumber + ''' ';
	  
  IF ISNULL(@answer,'') != '' 
    SET @MainQuery = @MainQuery +  ' AND EXISTS(SELECT fr.REGISTRANT_ID FROM AHOBPR.FORM_RESPONSE_ANSWER ra ' + 
                                    'JOIN AHOBPR.FORM_RESPONSE_QUESTION rq ON rq.FORM_RESPONSE_QUESTION_ID = ra.FORM_RESPONSE_QUESTION_ID  ' + 
                                    'JOIN AHOBPR.FORM_RESPONSE fr ON fr.FORM_RESPONSE_ID = rq.FORM_RESPONSE_ID  ' + 
                                    'JOIN AHOBPR.STD_FORM_ANSWER A1 ON A1.STD_FORM_ANSWER_ID = ra.STD_FORM_ANSWER_ID ' + 
                                    'JOIN AHOBPR.STD_FORM_QUESTION Q1 ON Q1.STD_FORM_QUESTION_ID = rq.STD_FORM_QUESTION_ID ' + 
                                    'JOIN AHOBPR.STD_FORM_QUESTION Q2 ON Q2.FORM_QUESTION_UNIQUE_ID = Q1.FORM_QUESTION_UNIQUE_ID ' + 
                                    'JOIN AHOBPR.STD_FORM_ANSWER A2 ON A2.STD_FORM_QUESTION_ID = Q2.STD_FORM_QUESTION_ID ' + 
                                    ' AND A1.BRP_FORM_ANSWER_ID = A2.BRP_FORM_ANSWER_ID ' + 
                                  ' WHERE A2.STD_FORM_ANSWER_ID IN (' + @answer + ') AND fr.REGISTRANT_ID = a.REGISTRANT_ID )';                                                    

  IF @formStartFromDate IS NOT NULL                     
     SET @MainQuery = @MainQuery + ' AND e.QUESTIONNAIRE_STARTED_DATE >= ''' + CONVERT(VARCHAR(20),@formStartFromDate,121) + '''';
     
  IF @formStartToDate IS NOT NULL                     
     SET @MainQuery = @MainQuery + ' AND e.QUESTIONNAIRE_STARTED_DATE <= ''' + CONVERT(VARCHAR(20),@formStartToDate,121) + '''';
    
  IF @formCompletionFromDate IS NOT NULL                     
     SET @MainQuery = @MainQuery + ' AND e.QUESTIONNAIRE_COMPLETED_DATE >= ''' + CONVERT(VARCHAR(20),@formCompletionFromDate,121) + '''';

  IF @formCompletionToDate IS NOT NULL                     
     SET @MainQuery = @MainQuery + ' AND e.QUESTIONNAIRE_COMPLETED_DATE <= ''' + CONVERT(VARCHAR(20),@formCompletionToDate,121) + '''';

  IF ISNULL(@subpopulationFlag,'') != ''
    SET @MainQuery = @MainQuery + ' AND EXISTS(SELECT * FROM AHOBPR.REGISTRANT_REGISTRY_FLAG subpop ' +
                                      ' WHERE subpop.STD_REGISTRY_FLAG_ID IN (' + @subpopulationFlag + ') ' +
                                        ' AND subpop.REGISTRANT_ID = a.REGISTRANT_ID ) ';

	IF ISNULL(@adminFlag,'') != ''
    SET @MainQuery = @MainQuery + ' AND EXISTS(SELECT * FROM AHOBPR.REGISTRANT_REGISTRY_FLAG adminflag ' +
                        ' WHERE adminflag.STD_REGISTRY_FLAG_ID IN (' + @adminFlag + ') ' +
                          ' AND adminflag.REGISTRANT_ID = a.REGISTRANT_ID ' +
                          ' AND adminflag.CREATED BETWEEN ''' + CONVERT(VARCHAR(20),@adminFlagFromDate,121) + 
                                                  ''' AND ''' + CONVERT(VARCHAR(20),@adminFlagToDate,121) + ''' ) ';  
                                              
  IF ISNULL(@deploymentAll,'') != '' 
    SET @MainQuery = @MainQuery + 'AND EXISTS (SELECT depl.REGISTRANT_ID FROM AHOBPR.REGISTRANT_DEPLOYMENT depl ' +
                                                  ' WHERE a.REGISTRANT_ID = depl.REGISTRANT_ID
                                                      AND depl.DEPLOYMENT_COUNTRY = ''' + @deploymentAll + ''') ' ;
                                                    
  IF ISNULL(@healthFactor,'') != ''
    SET @MainQuery = @MainQuery + ' AND EXISTS (SELECT re.REGISTRANT_ID FROM AHOBPR.REGISTRANT_VISIT re ' +
                                        ' JOIN AHOBPR.REGISTRANT_HEALTH_FACTOR hf ON re.REGISTRANT_VISIT_ID = hf.REGISTRANT_VISIT_ID ' +
                                       ' WHERE a.REGISTRANT_ID = re.REGISTRANT_ID AND re.EVALUATION_FLAG = ''Y'' ' +
                                       '   AND hf.STD_HEALTH_FACTOR_ID IN (' + @healthFactor + ') ) '

 
  IF ISNULL(@deploymentExposure,'') != '' 
    SET @MainQuery = @MainQuery + ' AND EXISTS(SELECT deploymentExposure.REGISTRANT_ID FROM AHOBPR.REGISTRANT_DEPLOYMENT deploymentExposure ' +
                                        '  JOIN AHOBPR.FORM_RESPONSE_QUESTION exposure ON deploymentExposure.REGISTRANT_DEPLOYMENT_ID = exposure.REGISTRANT_DEPLOYMENT_ID ' +
                                        '  JOIN AHOBPR.STD_FORM_QUESTION quest ON exposure.STD_FORM_QUESTION_ID = quest.STD_FORM_QUESTION_ID ' +
                                        ' WHERE quest.FORM_QUESTION_UNIQUE_ID = ''1.2.A'' ' +
                                        '   AND deploymentExposure.DEPLOYMENT_COUNTRY = ''' + @deploymentExposure + '''' + 
                                        '   AND deploymentExposure.REGISTRANT_ID = a.REGISTRANT_ID ) ';
                                                      
  
 -- The rest of the criteria will be applied to Main Query and With statememts 
  IF @status IS NOT NULL
    SET @RegCriteria = @RegCriteria + ' AND a.STD_REGISTRANT_STATUS_ID IN (' + @status + ')';

  IF ISNULL(@dutyStatus,'') > ''
    SET @RegCriteria = @RegCriteria + ' AND a.STD_SERVICE_STATUS_ID IN (' + @dutyStatus + ')';
          
  IF @healthConcerns != 'All'
    SET @RegCriteria = @RegCriteria + ' AND a.HEALTH_CONCERNS_RESPONSE = ''' + @healthConcerns + ''''; 
  
  IF @requestToBeSeen != 'All'
    SET @RegCriteria = @RegCriteria + ' AND a.REQUEST_SEEN_RESPONSE = ''' + @requestToBeSeen  + '''';
                                                                    
  IF ISNULL(@excludeDead,'N') = 'Y'
    SET @RegCriteria = @RegCriteria + ' AND a.DECEASED_FLAG = 0 ';
  
  IF ISNULL(@gender,'') != ''
    SET @RegCriteria = @RegCriteria + ' AND ''' + @gender + ''' = ISNULL(a.GENDER,''No Data Available'') ';
    
  IF ISNULL(@ethnicity,'') != ''
    SET @RegCriteria = @RegCriteria + ' AND ''' + @ethnicity + ''' = ISNULL(a.ETHNICITY,''No Data Available'') ';
  
  IF ISNULL(@separationType,0) != 0
    SET @RegCriteria = @RegCriteria + ' AND ''' + @separationType + ' = ISNULL(a.STD_SEPARATION_REASON_ID_CURRENT,0) ';

  IF @separationFromDate IS NOT NULL 
    SET @RegCriteria = @RegCriteria + ' AND a.SEPARATION_DATE_CURRENT >= ''' + CONVERT(VARCHAR(20),@separationFromDate,121) + '''';
  
  IF @separationToDate IS NOT NULL 
    SET @RegCriteria = @RegCriteria + ' AND  a.SEPARATION_DATE_CURRENT <= ''' + CONVERT(VARCHAR(20),@separationToDate,121) + '''';
  
  IF ISNULL(@race,'') != ''
    SET @RegCriteria = @RegCriteria + ' AND ''' + @race + ''' =  ISNULL(a.RACE ,''No Data Available'') ';
    
  IF ISNULL(@enrollmentStatus,'') != ''
    SET @RegCriteria = @RegCriteria + ' AND ''' + @enrollmentStatus + ''' = ISNULL(a.ENROLLMENT_STATUS,''No Data Available'') ';

  IF @AssignedLocationType IS NOT NULL
    BEGIN
      IF @AssignedLocationType = 'NLA' 
        SET @RegCriteria = @RegCriteria + ' AND a.STD_INSTITUTION_ID IS NULL ' ;
      ELSE
      IF @AssignedLocationType = 'VISN' 
        SET @RegCriteria = @RegCriteria + ' AND EXISTS(SELECT * FROM AHOBPR.VIEW_FACILITY_LOOKUP inst ' +
                                                             ' WHERE inst.ID = a.STD_INSTITUTION_ID ' + 
                                                             '   AND inst.VisnID = ''' + @AssignedLocationKey + ''') ';
      ELSE
      IF @AssignedLocationType = 'Base' 
        SET @RegCriteria = @RegCriteria + ' AND EXISTS(SELECT * FROM AHOBPR.VIEW_FACILITY_LOOKUP inst ' +
                                                      ' WHERE inst.ID = a.STD_INSTITUTION_ID ' +
                                                      '   AND inst.BASE_ID = ' + CONVERT(VARCHAR(20),@AssignedInstitutionId) + ') '
      ELSE
      IF @AssignedLocationType = 'Institution' 
        SET @RegCriteria = @RegCriteria + ' AND a.STD_INSTITUTION_ID = ' + CONVERT(VARCHAR(20),@AssignedInstitutionId);
   END;
   
  IF @EvaluatedLocationType IS NOT NULL
    BEGIN
      IF @EvaluatedLocationType = 'NLA' 
        SET @RegCriteria = @RegCriteria + ' AND NOT EXISTS( SELECT * FROM AHOBPR.REGISTRANT_VISIT eval ' +
                                                           ' WHERE eval.REGISTRANT_VISIT_ID = a.REGISTRANT_ID ' +
                                                           '   AND eval.EVALUATION_FLAG = ''Y'') ' ; 
      ELSE
      IF @EvaluatedLocationType = 'VISN' 
        SET @RegCriteria = @RegCriteria + ' AND EXISTS(SELECT * FROM AHOBPR.REGISTRANT_VISIT eval ' +
                                                       ' JOIN AHOBPR.VIEW_FACILITY_LOOKUP inst ON inst.ID = eval.STD_INSTITUTION_ID ' +
                                                      ' WHERE eval.REGISTRANT_VISIT_ID = a.REGISTRANT_ID ' +
                                                        ' AND eval.EVALUATION_FLAG = ''Y'' ' +
                                                        ' AND inst.VisnID = ''' + @EvaluatedLocationKey + ''') '; 
      ELSE
      IF @EvaluatedLocationType = 'Base' 
        SET @RegCriteria = @RegCriteria +  ' AND EXISTS(SELECT * FROM AHOBPR.REGISTRANT_VISIT eval ' +
                                                       ' JOIN AHOBPR.VIEW_FACILITY_LOOKUP inst ON inst.ID = eval.STD_INSTITUTION_ID ' +
                                                      ' WHERE eval.REGISTRANT_VISIT_ID = a.REGISTRANT_ID ' +
                                                        ' AND eval.EVALUATION_FLAG = ''Y'' ' +
                                                        ' AND inst.BASE_ID = ' + CONVERT(VARCHAR(20),@EvaluatedInstitutionId) + ') ';
      ELSE
      IF @EvaluatedLocationType = 'Institution' 
        SET @RegCriteria = @RegCriteria +  ' AND EXISTS(SELECT * FROM AHOBPR.REGISTRANT_VISIT eval ' +
                                                       ' JOIN AHOBPR.VIEW_FACILITY_LOOKUP inst ON inst.ID = eval.STD_INSTITUTION_ID ' +
                                                      ' WHERE eval.REGISTRANT_VISIT_ID = a.REGISTRANT_ID ' +
                                                        ' AND eval.EVALUATION_FLAG = ''Y'' ' +
                                                        ' AND eval.STD_INSTITUTION_ID = ' + CONVERT(VARCHAR(20),@EvaluatedInstitutionId)  + ') '; 
    END;
    
-- Add criteria and Close With Statement
  SET @DeployWith = @DeployWith + @RegCriteria + ' GROUP BY a.REGISTRANT_ID ) ' 
  SET @SubPopWith = @SubPopWith + @RegCriteria + ' GROUP BY a.REGISTRANT_ID ) ' 
  SET @AdminWith =  @AdminWith  + @RegCriteria + ' GROUP BY a.REGISTRANT_ID ) ' 

  SET @Query =  @SubPopWith +  @AdminWith + @DeployWith +  @MainQuery + @RegCriteria;            

  PRINT  @Query;
  
 EXECUTE (@Query);  
END;



-- Test Cases
-- EXECUTE [AHOBPR].[SP_ReportAdHocByIndividual] 
-- EXECUTE [AHOBPR].[SP_ReportAdHocByIndividual] @status = '6,7'
-- EXECUTE [AHOBPR].[SP_ReportAdHocByIndividual] @dutyStatus = '1,2'
-- EXECUTE [AHOBPR].[SP_ReportAdHocByIndividual] @formStartFromDateText  = '20010101'
-- EXECUTE [AHOBPR].[SP_ReportAdHocByIndividual] @formStartToDateText = '20141231'
-- EXECUTE [AHOBPR].[SP_ReportAdHocByIndividual] @healthFactor = '44,45,46,47' 
-- EXECUTE [AHOBPR].[SP_ReportAdHocByIndividual] @formCompletionFromDateText  = '20010101'
-- EXECUTE [AHOBPR].[SP_ReportAdHocByIndividual] @formCompletionToDateText  = '20141231'
-- EXECUTE [AHOBPR].[SP_ReportAdHocByIndividual] @healthConcerns = 'Yes'     , 
-- EXECUTE [AHOBPR].[SP_ReportAdHocByIndividual] @requestToBeSeen = 'Yes
-- EXECUTE [AHOBPR].[SP_ReportAdHocByIndividual] @subpopulationFlag = '64,65,66,67'
-- EXECUTE [AHOBPR].[SP_ReportAdHocByIndividual] @deploymentAll = 'Egypt'
-- EXECUTE [AHOBPR].[SP_ReportAdHocByIndividual] @deploymentExposure  = 'Iraq'
-- EXECUTE [AHOBPR].[SP_ReportAdHocByIndividual] @adminFlag = '59,60' ,@adminFlagFromDateText = '20150101' , @adminFlagToDateText = '20150201'
-- EXECUTE [AHOBPR].[SP_ReportAdHocByIndividual] @questionNumber = '5.4.A'  , @answer = '2221'
-- exec [AHOBPR].[SP_ReportAdHocByIndividual]
-- exec [AHOBPR].[SP_ReportAdHocByIndividual] @AssignedLocationType = 'NLA'
-- exec [AHOBPR].[SP_ReportAdHocByIndividual] @AssignedLocationType = 'VISN', @AssignedLocationKey = 'V16'
-- exec [AHOBPR].[SP_ReportAdHocByIndividual] @AssignedLocationType = 'Base', @AssignedLocationKey = '1000089'
-- exec [AHOBPR].[SP_ReportAdHocByIndividual] @AssignedLocationType = 'Institution', @AssignedLocationKey = '1000089'
-- exec [AHOBPR].[SP_ReportAdHocByIndividual]  @ethnicity = 'JEWISH'
-- exec [AHOBPR].[SP_ReportAdHocByIndividual] @EvaluatedLocationType = 'NLA'
-- exec [AHOBPR].[SP_ReportAdHocByIndividual] @EvaluatedLocationType = 'VISN', @EvaluatedLocationKey = 'V16'
-- exec [AHOBPR].[SP_ReportAdHocByIndividual] @EvaluatedLocationType = 'Base', @EvaluatedLocationKey = '1000089'
-- exec [AHOBPR].[SP_ReportAdHocByIndividual] @EvaluatedLocationType = 'Institution', @EvaluatedLocationKey = '1000089'

GO
PRINT 'ALTER PROCEDURE SP_ReportParticipation...';
GO
-- =============================================
-- Author:		Bruce Clegg
-- Create date: Dec 30, 2014
-- Description:	Used for the AHOBPR Participation Report       -- exec [AHOBPR].[SP_ReportParticipation] 
-- 20150330 - Bruce Clegg - Added VisnId value of NLA to find Null locations
-- 20150427 - Bruce Clegg - Change to LocationType and Key, for both Assigned and Evaluated Locations
-- 20150710 - Bruce Clegg - Rewrite using Dynamic SQL for Performance
-- 20150910 - Bruce Clegg - Removed selection for Question 1.2.A for Deployments
-- =============================================
ALTER PROCEDURE [AHOBPR].[SP_ReportParticipation]
       @StartDate DATETIME2(0) = NULL,  
       @EndDate   DATETIME2(0) = NULL,
       @AssignedLocationType  VARCHAR(20) = NULL,  -- 'VISN', 'Base', 'Institution', 'NLA' = No Location Assigned
       @AssignedLocationKey   VARCHAR(50) = NULL,  -- 'V16', 'V01'    1000109,  1000089
       @EvaluatedLocationType VARCHAR(20) = NULL,  -- 'VISN', 'Base', 'Institution', 'NLA' = No Location Assigned
       @EvaluatedLocationKey  VARCHAR(50) = NULL   -- 'V16', 'V01'    1000109,  1000089
AS
BEGIN

  SET @StartDate = ISNULL(@StartDate,'19900802'); -- Desert Storm start date
  SET @EndDate = ISNULL(@EndDate, GetDate());
  
  DECLARE @ConsentStatusCount INT = 0,
          @EligibleStatusCount INT = 0,
          @ParticipantStatusCount INT = 0,
          @HealthConcernsCount INT = 0,
          @NoHealthConcernsCount INT = 0,
          @RequestSeenCount INT = 0,
          @NoRequestSeenCount INT = 0,
          @DoNotKnowRequestSeenCount INT = 0,
          @VeteranServiceStatusCount INT = 0,
          @ActiveDutyServiceStatusCount INT = 0,
          @RetireeServiceStatusCount INT = 0,
          @Afgh911Count INT = 0,
          @Iraq911Count INT = 0,
          @SWA911Count INT = 0,
          @Pre911Count INT = 0,
          @Djibouti911Count INT = 0,
          @MedianDaysToParticipant DECIMAL(9,2) = 0;    
  
  DECLARE @AssignedInstitutionId INT = 0,
        @EvaluatedInstitutionId INT = 0;

  DECLARE @Query NVARCHAR(MAX);
  DECLARE @Criteria VARCHAR(2000) = '';

IF @AssignedLocationType IN ('Base','Institution') 
AND ISNUMERIC(@AssignedLocationKey) = 1
  SET @AssignedInstitutionId = CONVERT(INT,@AssignedLocationKey);
  
IF @EvaluatedLocationType IN ('Base','Institution') 
AND ISNUMERIC(@EvaluatedLocationKey) = 1
  SET @EvaluatedInstitutionId = CONVERT(INT,@EvaluatedLocationKey);

  --  Build Criteria string

  SET @Criteria = ' AND responseStat.QUESTIONNAIRE_COMPLETED_DATE BETWEEN ''' + CONVERT(VARCHAR(20),@StartDate,121) +
                                                                    ''' AND ''' + CONVERT(VARCHAR(20),@EndDate,121) + ''' ' ;
  IF @AssignedLocationType = 'NLA' 
    SET @Criteria = @Criteria + ' AND reg.STD_INSTITUTION_ID IS NULL ';

  IF @AssignedLocationType = 'VISN' 
    SET @Criteria = @Criteria + ' AND EXISTS(SELECT * FROM AHOBPR.VIEW_FACILITY_LOOKUP inst ' +
                                      ' WHERE inst.ID = reg.STD_INSTITUTION_ID ' +
                                         'AND inst.VisnID = ''' + @AssignedLocationKey + ''') ';
 	
  IF @AssignedLocationType = 'Base' 
    SET @Criteria = @Criteria + ' AND EXISTS(SELECT * FROM AHOBPR.VIEW_FACILITY_LOOKUP inst ' +
                                       'WHERE inst.ID = reg.STD_INSTITUTION_ID ' +
                                         'AND inst.BASE_ID = ' + CONVERT(VARCHAR(20),@AssignedInstitutionId) + ' ) ';
            	
  IF @AssignedLocationType = 'Institution' 
    SET @Criteria = @Criteria + ' AND reg.STD_INSTITUTION_ID = ' + CONVERT(VARCHAR(20),@AssignedInstitutionId) + ' '	
  
  IF @EvaluatedLocationType = 'NLA' 
    SET @Criteria = @Criteria + ' AND NOT EXISTS( SELECT * FROM AHOBPR.REGISTRANT_VISIT eval ' +
                                            'WHERE eval.REGISTRANT_ID = Reg.REGISTRANT_ID ' +
                                               'AND eval.EVALUATION_FLAG = ''Y'' ) ';
          
  IF @EvaluatedLocationType = 'VISN' 
    SET @Criteria = @Criteria + ' AND EXISTS(SELECT * FROM AHOBPR.REGISTRANT_VISIT eval ' +
                                               'JOIN AHOBPR.VIEW_FACILITY_LOOKUP inst ON inst.ID = eval.STD_INSTITUTION_ID ' +
                                             ' WHERE eval.REGISTRANT_ID = Reg.REGISTRANT_ID ' +
                                                'AND inst.VisnID = ''' +  @EvaluatedLocationKey + ''' ' +
                                                'AND eval.EVALUATION_FLAG = ''Y'') ' ; 
         	
  IF @EvaluatedLocationType = 'Base' 
    SET @Criteria = @Criteria + ' AND EXISTS(SELECT * FROM AHOBPR.REGISTRANT_VISIT eval ' +
                                               'JOIN AHOBPR.VIEW_FACILITY_LOOKUP inst ON inst.ID = eval.STD_INSTITUTION_ID ' +
                                             ' WHERE eval.REGISTRANT_ID = Reg.REGISTRANT_ID ' +
                                        ' AND inst.BASE_ID = ' + CONVERT(VARCHAR(20),@EvaluatedInstitutionId) + 
             	                          ' AND eval.EVALUATION_FLAG = ''Y'') '; 
  IF @EvaluatedLocationType = 'Institution' 
    SET @Criteria = @Criteria + ' AND EXISTS(SELECT * FROM AHOBPR.REGISTRANT_VISIT eval ' +
                                      ' WHERE eval.REGISTRANT_ID = Reg.REGISTRANT_ID '+
                                        ' AND eval.STD_INSTITUTION_ID = ' + CONVERT(VARCHAR(20),@EvaluatedInstitutionId) +
                                        ' AND eval.EVALUATION_FLAG = ''Y'') '; 

  --  @ParticipantStatusCount
  SET @Query = ' SELECT @ParticipantStatusCount = COUNT(*) ' +
                '  FROM [AHOBPR].[REGISTRANT] reg ' +
                '  JOIN AHOBPR.STD_REGISTRANT_STATUS stat ON stat.STD_REGISTRANT_STATUS_ID = reg.STD_REGISTRANT_STATUS_ID ' +
                '  JOIN AHOBPR.FORM_RESPONSE resp ON resp.REGISTRANT_ID = reg.REGISTRANT_ID ' +
                '  JOIN AHOBPR.FORM_RESPONSE_STATUS responseStat  ON responseStat.FORM_RESPONSE_ID = resp.FORM_RESPONSE_ID '  
              + @Criteria     
              + '  AND stat.REGISTRANT_STATUS = ''Participant'' '; 
                
  PRINT @Query;
  EXEC sp_executesql @Query,  N'@ParticipantStatusCount INT OUTPUT',    @ParticipantStatusCount OUTPUT;

   
    --  @HealthConcernsCount
  SET @Query = ' SELECT @HealthConcernsCount = COUNT(*) ' +
                '  FROM [AHOBPR].[REGISTRANT] reg ' +
                '  JOIN AHOBPR.STD_REGISTRANT_STATUS stat ON stat.STD_REGISTRANT_STATUS_ID = reg.STD_REGISTRANT_STATUS_ID ' +
                '  JOIN AHOBPR.FORM_RESPONSE resp ON resp.REGISTRANT_ID = reg.REGISTRANT_ID ' +
                '  JOIN AHOBPR.FORM_RESPONSE_STATUS responseStat  ON responseStat.FORM_RESPONSE_ID = resp.FORM_RESPONSE_ID '  
              + ' WHERE reg.HEALTH_CONCERNS_RESPONSE = ''Yes'' '
              + @Criteria; 
                
  PRINT @Query;
  EXEC sp_executesql @Query,  N'@HealthConcernsCount INT OUTPUT',    @HealthConcernsCount OUTPUT;

  --  @NoHealthConcernsCount
  SET @Query = ' SELECT @NoHealthConcernsCount = COUNT(*) ' +
                '  FROM [AHOBPR].[REGISTRANT] reg ' +
                '  JOIN AHOBPR.STD_REGISTRANT_STATUS stat ON stat.STD_REGISTRANT_STATUS_ID = reg.STD_REGISTRANT_STATUS_ID ' +
                '  JOIN AHOBPR.FORM_RESPONSE resp ON resp.REGISTRANT_ID = reg.REGISTRANT_ID ' +
                '  JOIN AHOBPR.FORM_RESPONSE_STATUS responseStat  ON responseStat.FORM_RESPONSE_ID = resp.FORM_RESPONSE_ID '  
              + ' WHERE reg.HEALTH_CONCERNS_RESPONSE IS NULL '
              + @Criteria; 
                
  PRINT @Query;
  EXEC sp_executesql @Query,  N'@NoHealthConcernsCount INT OUTPUT',    @NoHealthConcernsCount OUTPUT;
                           

  --  @RequestSeenCount
  SET @Query = ' SELECT @RequestSeenCount = COUNT(*) ' +
                '  FROM [AHOBPR].[REGISTRANT] reg ' +
                '  JOIN AHOBPR.STD_REGISTRANT_STATUS stat ON stat.STD_REGISTRANT_STATUS_ID = reg.STD_REGISTRANT_STATUS_ID ' +
                '  JOIN AHOBPR.FORM_RESPONSE resp ON resp.REGISTRANT_ID = reg.REGISTRANT_ID ' +
                '  JOIN AHOBPR.FORM_RESPONSE_STATUS responseStat  ON responseStat.FORM_RESPONSE_ID = resp.FORM_RESPONSE_ID '  
              + ' WHERE reg.REQUEST_SEEN_RESPONSE = ''Yes'' '
              + @Criteria;
                
  PRINT @Query;
  EXEC sp_executesql @Query,  N'@RequestSeenCount INT OUTPUT',    @RequestSeenCount OUTPUT;
                                                     
  --  @NoRequestSeenCount
  SET @Query = ' SELECT @NoRequestSeenCount = COUNT(*) ' +
                '  FROM [AHOBPR].[REGISTRANT] reg ' +
                '  JOIN AHOBPR.STD_REGISTRANT_STATUS stat ON stat.STD_REGISTRANT_STATUS_ID = reg.STD_REGISTRANT_STATUS_ID ' +
                '  JOIN AHOBPR.FORM_RESPONSE resp ON resp.REGISTRANT_ID = reg.REGISTRANT_ID ' +
                '  JOIN AHOBPR.FORM_RESPONSE_STATUS responseStat  ON responseStat.FORM_RESPONSE_ID = resp.FORM_RESPONSE_ID '  
              + ' WHERE reg.REQUEST_SEEN_RESPONSE = ''No'' '
              + @Criteria;
                
  PRINT @Query;
  EXEC sp_executesql @Query,  N'@NoRequestSeenCount INT OUTPUT',    @NoRequestSeenCount OUTPUT;
                                                                                         
  --  @DoNotKnowRequestSeenCount
  SET @Query = ' SELECT @DoNotKnowRequestSeenCount = COUNT(*) ' +
                '  FROM [AHOBPR].[REGISTRANT] reg ' +
                '  JOIN AHOBPR.STD_REGISTRANT_STATUS stat ON stat.STD_REGISTRANT_STATUS_ID = reg.STD_REGISTRANT_STATUS_ID ' +
                '  JOIN AHOBPR.FORM_RESPONSE resp ON resp.REGISTRANT_ID = reg.REGISTRANT_ID ' +
                '  JOIN AHOBPR.FORM_RESPONSE_STATUS responseStat  ON responseStat.FORM_RESPONSE_ID = resp.FORM_RESPONSE_ID '  
              + ' WHERE ISNULL(reg.REQUEST_SEEN_RESPONSE,'''') NOT IN (''Yes'',''No'') '
              + @Criteria;
                
  PRINT @Query;
  EXEC sp_executesql @Query,  N'@DoNotKnowRequestSeenCount INT OUTPUT', @DoNotKnowRequestSeenCount OUTPUT;      
                                                   
  --  @ActiveDutyServiceStatusCount
  SET @Query = ' SELECT @ActiveDutyServiceStatusCount = COUNT(*) ' +
                '  FROM [AHOBPR].[REGISTRANT] reg ' +
                '  JOIN AHOBPR.STD_REGISTRANT_STATUS stat ON stat.STD_REGISTRANT_STATUS_ID = reg.STD_REGISTRANT_STATUS_ID ' +
                '  JOIN AHOBPR.FORM_RESPONSE resp ON resp.REGISTRANT_ID = reg.REGISTRANT_ID ' +
                '  JOIN AHOBPR.FORM_RESPONSE_STATUS responseStat  ON responseStat.FORM_RESPONSE_ID = resp.FORM_RESPONSE_ID ' +
                '  JOIN AHOBPR.STD_SERVICE_STATUS svc  ON svc.STD_SERVICE_STATUS_ID = reg.STD_SERVICE_STATUS_ID '
              + ' WHERE svc.SERVICE_STATUS = ''Active Duty'' '
              + @Criteria;
                
  PRINT @Query;
  EXEC sp_executesql @Query,  N'@ActiveDutyServiceStatusCount INT OUTPUT', @ActiveDutyServiceStatusCount OUTPUT;        
         
   
  --  @VeteranServiceStatusCount
  SET @Query = ' SELECT @VeteranServiceStatusCount = COUNT(*) ' +
                '  FROM [AHOBPR].[REGISTRANT] reg ' +
                '  JOIN AHOBPR.STD_REGISTRANT_STATUS stat ON stat.STD_REGISTRANT_STATUS_ID = reg.STD_REGISTRANT_STATUS_ID ' +
                '  JOIN AHOBPR.FORM_RESPONSE resp ON resp.REGISTRANT_ID = reg.REGISTRANT_ID ' +
                '  JOIN AHOBPR.FORM_RESPONSE_STATUS responseStat  ON responseStat.FORM_RESPONSE_ID = resp.FORM_RESPONSE_ID ' +
                '  JOIN AHOBPR.STD_SERVICE_STATUS svc  ON svc.STD_SERVICE_STATUS_ID = reg.STD_SERVICE_STATUS_ID '
              + ' WHERE svc.SERVICE_STATUS = ''Separated'' '
              + @Criteria;
                
  PRINT @Query;
  EXEC sp_executesql @Query,  N'@VeteranServiceStatusCount INT OUTPUT', @VeteranServiceStatusCount OUTPUT;                    
 
  --  @RetireeServiceStatusCount
  SET @Query = ' SELECT @RetireeServiceStatusCount = COUNT(*) ' +
                '  FROM [AHOBPR].[REGISTRANT] reg ' +
                '  JOIN AHOBPR.STD_REGISTRANT_STATUS stat ON stat.STD_REGISTRANT_STATUS_ID = reg.STD_REGISTRANT_STATUS_ID ' +
                '  JOIN AHOBPR.FORM_RESPONSE resp ON resp.REGISTRANT_ID = reg.REGISTRANT_ID ' +
                '  JOIN AHOBPR.FORM_RESPONSE_STATUS responseStat  ON responseStat.FORM_RESPONSE_ID = resp.FORM_RESPONSE_ID ' +
                '  JOIN AHOBPR.STD_SERVICE_STATUS svc  ON svc.STD_SERVICE_STATUS_ID = reg.STD_SERVICE_STATUS_ID '
              + ' WHERE svc.SERVICE_STATUS = ''Retiree'' '
              + @Criteria;
                
  PRINT @Query;
  EXEC sp_executesql @Query,  N'@RetireeServiceStatusCount INT OUTPUT', @RetireeServiceStatusCount OUTPUT;                    
 
     
                                                         
-- Deployment Counts  
  SET @Query = 'WITH Deployments
  AS ( 
    SELECT 
        dply.REGISTRANT_ID ,
      CASE WHEN (SUM(CASE WHEN [DEPLOYMENT_COUNTRY] = ''Iraq'' 
                           AND [DEPLOYMENT_START_DATE] > ''20010911'' 
                           THEN 1 ELSE 0 END) )> 0 
           THEN 1 ELSE 0 END AS Iraq911Count,
      CASE WHEN (SUM(CASE WHEN [DEPLOYMENT_COUNTRY] = ''Afghanistan''  
                           AND [DEPLOYMENT_START_DATE] > ''20010911'' 
                          THEN 1 ELSE 0 END)) > 0
           THEN 1 ELSE 0 END AS Afgh911Count,              
      CASE WHEN (SUM(CASE WHEN [DEPLOYMENT_COUNTRY] = ''Djibouti''  
                           AND [DEPLOYMENT_START_DATE] > ''20010911'' 
                          THEN 1 ELSE 0 END)) > 0
           THEN 1 ELSE 0 END AS Djibouti911Count,              
      CASE WHEN (SUM(CASE WHEN [DEPLOYMENT_COUNTRY] IN (''Kuwait'',''Saudi Arabia'',''Bahrain'', ''Gulf of Aden'', ''Gulf of Oman'', ''Oman'', ''Qatar'', ''United Arab Emirates'', ''Persian Gulf'', ''Arabian Sea'', ''Red Sea'')
                           AND [DEPLOYMENT_START_DATE] > ''20010911'' 
                           THEN 1 ELSE 0 END)) > 0
           THEN 1 ELSE 0 END AS SWA911Count,
      CASE WHEN (SUM(CASE WHEN ([DEPLOYMENT_START_DATE] BETWEEN ''19900802'' AND ''20010911'' ) 
                           AND  [DEPLOYMENT_COUNTRY] IN (''Iraq'',''Kuwait'',''Saudi Arabia'',''Bahrain'', ''Gulf of Aden'', ''Gulf of Oman'', ''Oman'', ''Qatar'', ''United Arab Emirates'', ''Persian Gulf'', ''Arabian Sea'', ''Red Sea'')
                          THEN 1 ELSE 0 END)) > 0
           THEN 1 ELSE 0 END  AS Pre911Count
    FROM [AHOBPR].[REGISTRANT_DEPLOYMENT] dply
        JOIN AHOBPR.REGISTRANT Reg ON Reg.REGISTRANT_ID = dply.REGISTRANT_ID
        JOIN AHOBPR.FORM_RESPONSE resp ON resp.REGISTRANT_ID = dply.REGISTRANT_ID
        JOIN AHOBPR.FORM_RESPONSE_STATUS responseStat ON responseStat.FORM_RESPONSE_ID = resp.FORM_RESPONSE_ID   
	     WHERE 1 = 1 ' +
	     @Criteria +                
    ' GROUP BY dply.REGISTRANT_ID 
     )
     SELECT @Afgh911Count =  SUM(ISNULL(Deployments.Afgh911Count,0))  
           ,@Iraq911Count =  SUM(ISNULL(Deployments.Iraq911Count,0))
           ,@SWA911Count = SUM(ISNULL(Deployments.SWA911Count,0))
           ,@Pre911Count =   SUM(ISNULL(Deployments.Pre911Count,0))
           ,@Djibouti911Count = SUM(ISNULL(Deployments.Djibouti911Count,0))
      FROM Deployments' ;
      
  PRINT @Query;
  EXEC sp_executesql @Query,  
       N'@Afgh911Count INT OUTPUT, @Iraq911Count INT OUTPUT, @SWA911Count INT OUTPUT, @Pre911Count INT OUTPUT, @Djibouti911Count INT OUTPUT'
      , @Afgh911Count OUTPUT, @Iraq911Count OUTPUT, @SWA911Count OUTPUT, @Pre911Count OUTPUT, @Djibouti911Count OUTPUT;            

  
     CREATE TABLE #DaysToParticipant(DAYS INT);
   
    -- Median Days from Eligible to Participant Status    
    -- Note that only Registrants with Both values and DateParticipant > DateEligible will be counted for either Numerator or Denominator

    SET @Query = 'WITH eligible 
      AS ( 
          SELECT [REGISTRANT_ID],MIN([CREATED]) AS DateEligible
            FROM [AHOBPR].[PROCESS_METRICS]
           WHERE [PROCESS_METRICS_VALUE] = ''Eligible''
           GROUP BY [REGISTRANT_ID]
         ),
      participant 
      AS (
         SELECT [REGISTRANT_ID],MAX([CREATED]) AS DateParticipant
           FROM [AHOBPR].[PROCESS_METRICS]
          WHERE [PROCESS_METRICS_VALUE] = ''Participant''   
           GROUP BY [REGISTRANT_ID]
          )
     INSERT INTO #DaysToParticipant(DAYS)
    SELECT 
      CASE WHEN DateParticipant > DateEligible THEN DATEDIFF(Day,DateEligible,DateParticipant) ELSE 0 END AS DaysDifference
      FROM [AHOBPR].[REGISTRANT] reg
      JOIN eligible ON reg.REGISTRANT_ID = eligible.REGISTRANT_ID
      JOIN participant ON reg.REGISTRANT_ID = participant.REGISTRANT_ID   
      JOIN AHOBPR.FORM_RESPONSE resp ON resp.REGISTRANT_ID = reg.REGISTRANT_ID
      JOIN AHOBPR.FORM_RESPONSE_STATUS responseStat ON responseStat.FORM_RESPONSE_ID = resp.FORM_RESPONSE_ID ' +
    'WHERE 1 = 1 '  + @Criteria +  
   ' ORDER BY DaysDifference';

  PRINT @Query;
  EXECUTE(@Query);
     
    WITH Counts AS
    (
       SELECT c=COUNT(*)
       FROM #DaysToParticipant
    )
    SELECT @MedianDaysToParticipant = AVG(0. + DAYS)
    FROM Counts a
    CROSS APPLY
    (
       SELECT TOP(((a.c - 1) / 2) + (1 + (1 - a.c % 2)))
          DAYS, r=ROW_NUMBER() OVER (ORDER BY DAYS)
       FROM #DaysToParticipant b
       ORDER BY DAYS
    ) p
    WHERE r BETWEEN ((a.c - 1) / 2) + 1 AND (((a.c - 1) / 2) + (1 + (1 - a.c % 2))); 
         
    DROP  TABLE #DaysToParticipant;                   
 
 -- Return Results
   SELECT ISNULL(@ParticipantStatusCount,0) AS ParticipantStatusCount,
          ISNULL(@HealthConcernsCount,0) AS ReportedHealthConcernsCount,
          ISNULL(@NoHealthConcernsCount,0) AS  NoReportedHealthConcernsCount,
          ISNULL(@RequestSeenCount,0) AS RequestSeenCount,
          ISNULL(@NoRequestSeenCount,0) AS NoRequestSeenCount,
          ISNULL(@DoNotKnowRequestSeenCount,0) AS DoNotKnowRequestSeenCount,
          ISNULL(@VeteranServiceStatusCount,0) AS VeteranServiceStatusCount ,
          ISNULL(@ActiveDutyServiceStatusCount,0) AS  ActiveDutyServiceStatusCount,
          ISNULL(@RetireeServiceStatusCount,0) AS RetireeServiceStatusCount,
          ISNULL(@Afgh911Count,0) AS  Afgh911Count,
          ISNULL(@Iraq911Count,0) AS Iraq911Count,
          ISNULL(@SWA911Count,0) AS  SouthWestAsia911Count,
          ISNULL(@Pre911Count,0) AS  Pre911Count,
          ISNULL(@MedianDaysToParticipant,0.0) AS MedianDaysToParticipant,
          ISNULL(@Djibouti911Count,0) AS Djibouti911Count ;

END

-- exec [AHOBPR].[SP_ReportParticipation]
-- exec [AHOBPR].[SP_ReportParticipation] @AssignedLocationType = 'NLA'
-- exec [AHOBPR].[SP_ReportParticipation] @AssignedLocationType = 'VISN', @AssignedLocationKey = 'V16'
-- exec [AHOBPR].[SP_ReportParticipation] @AssignedLocationType = 'Base', @AssignedLocationKey = '1000089'
-- exec [AHOBPR].[SP_ReportParticipation] @AssignedLocationType = 'Institution', @AssignedLocationKey = '1000089'
-- exec [AHOBPR].[SP_ReportParticipation]
-- exec [AHOBPR].[SP_ReportParticipation] @EvaluatedLocationType = 'NLA'
-- exec [AHOBPR].[SP_ReportParticipation] @EvaluatedLocationType = 'VISN', @EvaluatedLocationKey = 'V16'
-- exec [AHOBPR].[SP_ReportParticipation] @EvaluatedLocationType = 'Base', @EvaluatedLocationKey = '1000089'
-- exec [AHOBPR].[SP_ReportParticipation] @EvaluatedLocationType = 'Institution', @EvaluatedLocationKey = '1000089'
GO

PRINT 'ALTER PROCEDURE SP_ReportParticipantDetails...';
GO
-- =============================================
-- Author:		Bruce Clegg
-- Create date: Jan 5, 2015
-- Description:	Used for the AHOBPR Participation Report Drilldown  -- exec [AHOBPR].[SP_ReportParticipantDetails]    (More options at bottom)
-- 20150330 - Bruce Clegg - Added VisnId value of NLA to find Null locations
-- 20150427 - Bruce Clegg - Change to LocationType and Key, for both Assigned and Evaluated Locations
-- 20150504 - Bruce Clegg - Fix @RequestSeen selection logic to match Participation Report
-- 20150710 - Bruce Clegg - Rewrite using Dynamic SQL for Performance
-- 20150910 - Bruce Clegg - Removed selection for Question 1.2.A for Deployments
-- =============================================
ALTER PROCEDURE [AHOBPR].[SP_ReportParticipantDetails]
       @StartDate DATETIME2(0) = NULL,  
       @EndDate DATETIME2(0) = NULL,
       @RegistrantStatus  VARCHAR(50) = '',  -- 'Participant', 'Review - Not Eligible', 'Eligible', 'No Consent', 'Consent', 'Review'
       @SvcStatus  VARCHAR(50) = '',         -- 'Veteran', 'Retiree', 'Active Duty'
       @RequestSeen  VARCHAR(50) = '',       -- 'Yes', 'No', 'Unknown'
       @HealthConcern  VARCHAR(50) = '',     -- 'Yes', 'No'
       @DeploymentGroup VARCHAR(50) = '',    -- 'Afgh911', 'Iraq911', 'SouthwestAsiaAfter911', 'Pre911',  'DjiboutiAfter911'
       @AssignedLocationType  VARCHAR(20) = NULL,  -- 'VISN', 'Base', 'Institution', 'NLA' = No Location Assigned
       @AssignedLocationKey   VARCHAR(50) = NULL,  -- 'V16', 'V01'    1000109,  1000089
       @EvaluatedLocationType VARCHAR(20) = NULL,  -- 'VISN', 'Base', 'Institution', 'NLA' = No Location Assigned
       @EvaluatedLocationKey  VARCHAR(50) = NULL   -- 'V16', 'V01'    1000109,  1000089
AS
BEGIN

  SET @StartDate = ISNULL(@StartDate,'19900802');
  SET @EndDate = ISNULL(@EndDate, GetDate());
  SET @EndDate = dbo.AddMaxTime(@EndDate);  -- Adds 23:59:59 to criteria parameter

  DECLARE @AssignedInstitutionId INT = 0,
          @EvaluatedInstitutionId INT = 0;
  DECLARE @Query NVARCHAR(MAX);

  IF @AssignedLocationType IN ('Base','Institution') 
  AND ISNUMERIC(@AssignedLocationKey) = 1
    SET @AssignedInstitutionId = CONVERT(INT,@AssignedLocationKey);
  
  IF @EvaluatedLocationType IN ('Base','Institution') 
  AND ISNUMERIC(@EvaluatedLocationKey) = 1
    SET @EvaluatedInstitutionId = CONVERT(INT,@EvaluatedLocationKey);

  --Temp table for Deployments
  CREATE TABLE #Deployments
  (
	   [REGISTRANT_ID] INT
  ); 
  
  -- INSERT Registrants if they are in the Deployment Group
  SET @Query = ' INSERT INTO #Deployments([REGISTRANT_ID]) ' +
               ' SELECT DISTINCT deployment.REGISTRANT_ID ' +
               '   FROM AHOBPR.REGISTRANT_DEPLOYMENT deployment  ' +
               '  WHERE 1 = 1 ';
  IF @DeploymentGroup = 'Afgh911'    
     SET @Query =  @Query + ' AND [DEPLOYMENT_COUNTRY] = ''Afghanistan'' ' + 
                            ' AND [DEPLOYMENT_START_DATE] > ''20010911'' ';
  ELSE
  IF @DeploymentGroup = 'Iraq911'
      SET @Query =  @Query + ' AND [DEPLOYMENT_COUNTRY] = ''Iraq'' ' + 
                              'AND [DEPLOYMENT_START_DATE] > ''20010911'' '; 
  ELSE
  IF @DeploymentGroup = 'DjiboutiAfter911'
      SET @Query =  @Query + ' AND [DEPLOYMENT_COUNTRY] = ''Djibouti'' ' + 
                              'AND [DEPLOYMENT_START_DATE] > ''20010911'' '; 
  ELSE
  IF @DeploymentGroup = 'SouthwestAsiaAfter911'
      SET @Query =  @Query + ' AND [DEPLOYMENT_COUNTRY] IN (''Kuwait'',''Saudi Arabia'',''Bahrain'', ''Gulf of Aden'', ''Gulf of Oman'', ''Oman'', ''Qatar'', ''United Arab Emirates'', ''Persian Gulf'', ''Arabian Sea'', ''Red Sea'') ' +
                             'AND [DEPLOYMENT_START_DATE] > ''20010911''  ';
  ELSE
  IF @DeploymentGroup = 'Pre911'
      SET @Query =  @Query + ' AND [DEPLOYMENT_COUNTRY] IN (''Iraq'',''Kuwait'',''Saudi Arabia'',''Bahrain'', ''Gulf of Aden'', ''Gulf of Oman'', ''Oman'', ''Qatar'', ''United Arab Emirates'', ''Persian Gulf'', ''Arabian Sea'', ''Red Sea'') ' +     
                             ' AND [DEPLOYMENT_START_DATE] BETWEEN ''19900802'' AND ''20010911''  ';
  IF ISNULL(@DeploymentGroup,'') > ''
    EXECUTE(@Query);
   
  SET @Query = '
    WITH eligible 
      AS ( 
          SELECT [REGISTRANT_ID],MIN([CREATED]) AS DateEligible
            FROM [AHOBPR].[PROCESS_METRICS]
           WHERE [PROCESS_METRICS_VALUE] IN (''Eligible'',''Reviewed - Eligible'')
           GROUP BY [REGISTRANT_ID]
         ),
      participant 
      AS (
         SELECT [REGISTRANT_ID], MAX([CREATED]) AS DateParticipant
           FROM [AHOBPR].[PROCESS_METRICS]
          WHERE [PROCESS_METRICS_VALUE] = ''Participant''   
          GROUP BY [REGISTRANT_ID]
          )        
  	SELECT reg.REGISTRANT_ID
		     ,(reg.LAST_NAME + '', '' + reg.FIRST_NAME) AS Name
		     ,stat.REGISTRANT_STATUS
		     ,svc.SERVICE_STATUS
		     ,QUESTIONNAIRE_COMPLETED_DATE
		     ,ISNULL(reg.REQUEST_SEEN_RESPONSE, ''Unknown'') AS REQUEST_SEEN_RESPONSE
		     ,ISNULL(reg.HEALTH_CONCERNS_RESPONSE,''No'') AS HEALTH_CONCERNS_RESPONSE
		     ,ISNULL(inst.Name,'''') AS InstitutionName
		     ,CONVERT(VARCHAR(10), responseStat.QUESTIONNAIRE_COMPLETED_DATE, 101) AS QUESTIONNAIRE_COMPLETED_Display
		     ,CASE WHEN DateParticipant > DateEligible THEN DATEDIFF(Day,DateEligible,DateParticipant) ELSE 0 END AS DaysEligibleToParticipant
         ,reg.BIRTH_DATE AS BirthDate
         ,CONVERT(VARCHAR(10), reg.BIRTH_DATE, 101) AS BirthDateDisplay
	  FROM AHOBPR.REGISTRANT reg
		JOIN AHOBPR.STD_REGISTRANT_STATUS stat 
		  ON reg.STD_REGISTRANT_STATUS_ID = stat.STD_REGISTRANT_STATUS_ID
		JOIN AHOBPR.STD_SERVICE_STATUS svc
		  ON svc.STD_SERVICE_STATUS_ID = reg.STD_SERVICE_STATUS_ID
		JOIN AHOBPR.FORM_RESPONSE resp 
		  ON resp.REGISTRANT_ID = reg.REGISTRANT_ID
		JOIN AHOBPR.FORM_RESPONSE_STATUS responseStat
		  ON responseStat.FORM_RESPONSE_ID = resp.FORM_RESPONSE_ID
    JOIN eligible 
      ON reg.REGISTRANT_ID = eligible.REGISTRANT_ID
    JOIN participant 
      ON reg.REGISTRANT_ID = participant.REGISTRANT_ID   		  
		LEFT JOIN AHOBPR.VIEW_FACILITY_LOOKUP inst
		  ON inst.ID = reg.STD_INSTITUTION_ID
 	 WHERE responseStat.QUESTIONNAIRE_COMPLETED_DATE BETWEEN ''' +CONVERT(VARCHAR(20), @StartDate,121) + 
                                                   ''' AND ''' + CONVERT(VARCHAR(20), @EndDate,121) + ''' ';

  IF ISNULL(@RegistrantStatus,'') > ''
	  SET @Query = @Query + ' AND stat.REGISTRANT_STATUS = ''' + @RegistrantStatus + ''' ';

  IF ISNULL(@SvcStatus,'') > ''
	  SET @Query = @Query + ' AND svc.SERVICE_STATUS = ''' +  @SvcStatus + ''' ';

  IF ISNULL(@HealthConcern,'') > ''
	  SET @Query = @Query + ' AND ISNULL(reg.HEALTH_CONCERNS_RESPONSE,''No'') = ''' +  @HealthConcern + ''' ';

  IF ISNULL(@RequestSeen,'') > ''
	  SET @Query = @Query + 'AND ISNULL(reg.REQUEST_SEEN_RESPONSE, ''Unknown'') = ''' +  @RequestSeen + ''' ';

  IF ISNULL(@DeploymentGroup,'') > ''
	  SET @Query = @Query + 'AND EXISTS(SELECT * FROM #Deployments WHERE #Deployments.REGISTRANT_ID = reg.REGISTRANT_ID) ';

  IF @AssignedLocationType = 'NLA' 
    SET @Query = @Query + ' AND reg.STD_INSTITUTION_ID IS NULL ' ;

  IF @AssignedLocationType = 'VISN' 
    SET @Query = @Query + ' AND EXISTS(SELECT * FROM AHOBPR.VIEW_FACILITY_LOOKUP inst ' +
                                                          ' WHERE inst.ID = reg.STD_INSTITUTION_ID ' + 
                                                          '   AND inst.VisnID = ''' + @AssignedLocationKey + ''') ';
 
  IF @AssignedLocationType = 'Base' 
    SET @Query = @Query + ' AND EXISTS(SELECT * FROM AHOBPR.VIEW_FACILITY_LOOKUP inst ' +
                                                  ' WHERE inst.ID = reg.STD_INSTITUTION_ID ' +
                                                  '   AND inst.BASE_ID = ' + CONVERT(VARCHAR(20),@AssignedInstitutionId) + ') '
 
  IF @AssignedLocationType = 'Institution' 
    SET @Query = @Query + ' AND reg.STD_INSTITUTION_ID = ' + CONVERT(VARCHAR(20),@AssignedInstitutionId);

  IF @EvaluatedLocationType = 'NLA' 
    SET @Query = @Query + ' AND NOT EXISTS( SELECT * FROM AHOBPR.REGISTRANT_VISIT eval ' +
                                                        ' WHERE eval.REGISTRANT_VISIT_ID = reg.REGISTRANT_ID ' +
                                                        '   AND eval.EVALUATION_FLAG = ''Y'') ' ; 
 
  IF @EvaluatedLocationType = 'VISN' 
    SET @Query = @Query + ' AND EXISTS(SELECT * FROM AHOBPR.REGISTRANT_VISIT eval ' +
                                                    ' JOIN AHOBPR.VIEW_FACILITY_LOOKUP inst ON inst.ID = eval.STD_INSTITUTION_ID ' +
                                                  ' WHERE eval.REGISTRANT_VISIT_ID = reg.REGISTRANT_ID ' +
                                                    ' AND eval.EVALUATION_FLAG = ''Y'' ' +
                                                    ' AND inst.VisnID = ''' + @EvaluatedLocationKey + ''') '; 
 
  IF @EvaluatedLocationType = 'Base' 
    SET @Query = @Query +  ' AND EXISTS(SELECT * FROM AHOBPR.REGISTRANT_VISIT eval ' +
                                                    ' JOIN AHOBPR.VIEW_FACILITY_LOOKUP inst ON inst.ID = eval.STD_INSTITUTION_ID ' +
                                                  ' WHERE eval.REGISTRANT_VISIT_ID = reg.REGISTRANT_ID ' +
                                                    ' AND eval.EVALUATION_FLAG = ''Y'' ' +
                                                    ' AND inst.BASE_ID = ' + CONVERT(VARCHAR(20),@EvaluatedInstitutionId) + ') ';
  
  IF @EvaluatedLocationType = 'Institution' 
    SET @Query = @Query +  ' AND EXISTS(SELECT * FROM AHOBPR.REGISTRANT_VISIT eval ' +
                                                    ' JOIN AHOBPR.VIEW_FACILITY_LOOKUP inst ON inst.ID = eval.STD_INSTITUTION_ID ' +
                                                  ' WHERE eval.REGISTRANT_VISIT_ID = reg.REGISTRANT_ID ' +
                                                    ' AND eval.EVALUATION_FLAG = ''Y'' ' +
                                                    ' AND eval.STD_INSTITUTION_ID = ' + CONVERT(VARCHAR(20),@EvaluatedInstitutionId)  + ') '; 

  SET @Query = @Query + ' ORDER BY responseStat.QUESTIONNAIRE_COMPLETED_DATE ';

  PRINT @Query;
  
  EXECUTE(@Query);
      
  DROP TABLE #Deployments;                                                                                                       
END

-- exec [AHOBPR].[SP_ReportParticipantDetails] 
-- exec [AHOBPR].[SP_ReportParticipantDetails] '20140101', '20150101', NULL, NULL, 'Unknown'
-- exec [AHOBPR].[SP_ReportParticipantDetails] NULL, NULL, 'Participant'
-- exec [AHOBPR].[SP_ReportParticipantDetails] NULL, NULL, NULL, 'Separated'
-- exec [AHOBPR].[SP_ReportParticipantDetails] NULL, NULL, NULL, NULL, 'Yes'
-- exec [AHOBPR].[SP_ReportParticipantDetails] NULL, NULL, NULL, NULL, 'No'
-- exec [AHOBPR].[SP_ReportParticipantDetails] NULL, NULL, NULL, NULL, 'Unknown'
-- exec [AHOBPR].[SP_ReportParticipantDetails] NULL, NULL, NULL, NULL, NULL, 'Yes'
-- exec [AHOBPR].[SP_ReportParticipantDetails] NULL, NULL, NULL, NULL, NULL, 'No'
-- exec [AHOBPR].[SP_ReportParticipantDetails] NULL, NULL, NULL, NULL, NULL, NULL, 'Afgh911'
-- exec [AHOBPR].[SP_ReportParticipantDetails] NULL, NULL, NULL, NULL, NULL, NULL, 'Iraq911'
-- exec [AHOBPR].[SP_ReportParticipantDetails] NULL, NULL, NULL, NULL, NULL, NULL, 'SouthwestAsiaAfter911'
-- exec [AHOBPR].[SP_ReportParticipantDetails] NULL, NULL, NULL, NULL, NULL, NULL, 'DjiboutiAfter911'
-- exec [AHOBPR].[SP_ReportParticipantDetails] NULL, NULL, NULL, NULL, NULL, NULL, 'Pre911'

-- exec [AHOBPR].[SP_ReportParticipantDetails]
-- exec [AHOBPR].[SP_ReportParticipantDetails] @AssignedLocationType = 'NLA'
-- exec [AHOBPR].[SP_ReportParticipantDetails] @AssignedLocationType = 'VISN', @AssignedLocationKey = 'V16'
-- exec [AHOBPR].[SP_ReportParticipantDetails] @AssignedLocationType = 'Base', @AssignedLocationKey = '1000089'
-- exec [AHOBPR].[SP_ReportParticipantDetails] @AssignedLocationType = 'Institution', @AssignedLocationKey = '1000089'
-- exec [AHOBPR].[SP_ReportParticipantDetails]
-- exec [AHOBPR].[SP_ReportParticipantDetails] @EvaluatedLocationType = 'NLA'
-- exec [AHOBPR].[SP_ReportParticipantDetails] @EvaluatedLocationType = 'VISN', @EvaluatedLocationKey = 'V16'
-- exec [AHOBPR].[SP_ReportParticipantDetails] @EvaluatedLocationType = 'Base', @EvaluatedLocationKey = '1000089'
-- exec [AHOBPR].[SP_ReportParticipantDetails] @EvaluatedLocationType = 'Institution', @EvaluatedLocationKey = '1000089'
GO

INSERT INTO [dbo].[DatabaseChangeLog]
           ([MajorReleaseNumber]
           ,[MinorReleaseNumber]
           ,[PointReleaseNumber]
           ,[Hotfix]
           ,[ScriptName]
           ,[ScriptDescription]
           ,[TargetRegistry]
           ,[DateApplied])
     VALUES
           ('$(MajorReleaseNumber)'
           ,'$(MinorReleaseNumber)'
           ,'$(PointReleaseNumber)'
           ,0
           ,'Registry.AHOBPR.Database.Build.1.0.51.sql'
		       ,'Registry.AHOBPR.Database.Build.1.0.51.sql Script'
           ,'AHOBPR'
           ,GETDATE());
GO

PRINT '';
PRINT 'Registry.AHOBPR.Database.Build.1.0.51.sql Script Complete.';
PRINT '';
PRINT 'Completed ' + CAST(GETDATE() AS varchar(20)) + '.';

---------------------------------
-- END OF GENERATED DEPLOYMENT --
---------------------------------
